#' print.CAISEr
#'
#' S3 method for printing _CAISEr_ objects (the output of
#' [run_experiment()]).
#'
#' @param x list object of class _CAISEr_
#'          (generated by [run_experiment()])
#' @param ... other parameters to be passed down to specific
#'            summary functions (currently unused)
#' @param echo logical flag: should the print method actually print to screen?
#' @param digits the minimum number of significant digits to be used.
#'               See [print.default()].
#' @param right logical, indicating whether or not strings should be
#'              right-aligned.
#' @param breakrows logical, indicating whether to "widen" the output table by
#'                  placing the bottom half to the right of the top half.
#'
#' @examples
#' # Example using four dummy algorithms and 100 dummy instances.
#' # See [dummyalgo()] and [dummyinstance()] for details.
#' # Generating 4 dummy algorithms here, with means 15, 10, 30, 15 and standard
#' # deviations 2, 4, 6, 8.
#' algorithms <- mapply(FUN = function(i, m, s){
#'   list(FUN   = "dummyalgo",
#'        alias = paste0("algo", i),
#'        distribution.fun  = "rnorm",
#'        distribution.pars = list(mean = m, sd = s))},
#'   i = c(alg1 = 1, alg2 = 2, alg3 = 3, alg4 = 4),
#'   m = c(15, 10, 30, 15),
#'   s = c(2, 4, 6, 8),
#'   SIMPLIFY = FALSE)
#'
#' # Generate 100 dummy instances with centered exponential distributions
#' instances <- lapply(1:100,
#'                     function(i) {rate <- runif(1, 1, 10)
#'                                  list(FUN   = "dummyinstance",
#'                                       alias = paste0("Inst.", i),
#'                                       distr = "rexp", rate = rate,
#'                                       bias  = -1 / rate)})
#'
#' my.results <- run_experiment(instances, algorithms,
#'                              d = 1, se.max = .1,
#'                              power = .9, sig.level = .05,
#'                              power.target = "mean",
#'                              dif = "perc", comparisons = "all.vs.all",
#'                              seed = 1234, ncpus = 1)
#' my.results
#'
#'
#' @return data frame object containing the summary table (invisibly)
#'
#' @method print CAISEr
#' @export
#'
print.CAISEr <- function(x, ...,
                         echo = TRUE,
                         digits = 4,
                         right = TRUE,
                         breakrows = FALSE)
{

  # Error checking
  assertthat::assert_that("CAISEr" %in% class(x))

  # ===========================================================================

  my.table <- x$data.summary

  if (breakrows){
    ninst      <- nrow(my.table)
    breakpoint <- ceiling(ninst / 2)
    tophalf    <- my.table[1:breakpoint, ]
    bottomhalf <- my.table[(breakpoint + 1):ninst, ]
    if(nrow(tophalf) > nrow(bottomhalf)) bottomhalf <- rbind(bottomhalf,
                                                             rep(NA, 5))
    my.table   <- cbind(tophalf,
                        `|` = rep("|", breakpoint),
                        bottomhalf)
  }

  # Print summary
  if(echo){
    cat("#====================================")
    cat("\n Summary table of CAISEr object\n")
    print.data.frame(my.table[, 1:(ncol(my.table) - 2)],
                     digits = digits, right = right,
                     quote = FALSE, row.names = FALSE)
    cat("\n#====================================")
  }

  invisible(my.table)
}
