#' @title Create BayesTools ensemble summary tables
#'
#' @description Creates estimate summaries based on posterior
#' distributions created by [mix_posteriors], inference summaries
#' based on inference created by [ensemble_inference], or ensemble
#' summary/diagnostics based on a list of [models_inference] models.
#'
#' @param samples posterior samples created by [mix_posteriors]
#' @param inference model inference created by [ensemble_inference]
#' @param parameters character vector of parameters (or a
#' named list with of character vectors for summary and
#' diagnostics tables) specifying the parameters
#' (and their grouping) for the summary table
#' @param models list of [models_inference] model objects,
#' each of which containing a list of \code{priors}
#' and \code{inference} object, The \code{inference} must be a
#' named list with information about the model: model number
#' \code{m_number}, marginal likelihood \code{marglik}, prior and
#' posterior probability \code{prior_prob} and \code{post_prob},
#' inclusion Bayes factor \code{inclusion_BF}, and fit summary
#' generated by [runjags_estimates_table] for the diagnostics
#' table
#' @param probs quantiles for parameter estimates
#' @param logBF whether the Bayes factor should be on log scale
#' @param BF01 whether the Bayes factor should be inverted
#' @param short_name whether the prior distribution names should be
#' shortened. Defaults to \code{FALSE}.
#' @param remove_spike_0 whether prior distributions equal to spike
#' at 0 should be removed from the \code{prior_list}
#' @param title title to be added to the table
#' @param footnotes footnotes to be added to the table
#' @param warnings warnings to be added to the table
#'
#'
#' @return \code{ensemble_estimates_table} returns a table with the
#' model-averaged estimates, \code{ensemble_inference_table} returns
#' a table with the prior and posterior probabilities and inclusion
#' Bayes factors, \code{ensemble_summary_table} returns a table with
#' overview of the models included in the ensemble, and
#' \code{ensemble_diagnostics_table} returns an overview of the MCMC
#' diagnostics for the models included in the ensemble. All of the
#' tables are objects of class 'BayesTools_table'.
#'
#' @export ensemble_estimates_table
#' @export ensemble_inference_table
#' @export ensemble_summary_table
#' @export ensemble_diagnostics_table
#' @name BayesTools_ensemble_tables
#'
#' @seealso [ensemble_inference] [mix_posteriors] [BayesTools_model_tables]
NULL

#' @rdname BayesTools_ensemble_tables
ensemble_estimates_table <- function(samples, parameters, probs = c(0.025, 0.95), title = NULL, footnotes = NULL, warnings = NULL){

  # check input
  check_char(parameters, "parameters", check_length = 0)
  check_list(samples, "samples", check_names = parameters, all_objects = TRUE, allow_other = TRUE)
  check_real(probs, "probs", lower = 0, upper = 1, check_length = 0, allow_NULL = TRUE)
  check_char(title, "title", allow_NULL = TRUE)
  check_char(footnotes, "footnotes", check_length = 0, allow_NULL = TRUE)
  check_char(warnings, "warnings", check_length = 0, allow_NULL = TRUE)

  # extract values
  estimates_table <- NULL
  for(parameter in parameters){

    if(is.matrix(samples[[parameter]])){

      par_summary <- cbind(
        "Mean"   = apply(samples[[parameter]], 2, mean),
        "Median" = apply(samples[[parameter]], 2, stats::median)
      )
      for(i in seq_along(probs)){
        par_summary <- cbind(par_summary, apply(samples[[parameter]], 2, stats::quantile, probs = probs[i]))
        colnames(par_summary)[ncol(par_summary)] <- probs[i]
      }
      rownames(par_summary) <- colnames(samples[[parameter]])
      estimates_table       <- rbind(estimates_table, par_summary)

    }else if(is.numeric(samples[[parameter]])){

      par_summary <- c(
        "Mean"   = mean(samples[[parameter]]),
        "Median" = stats::median((samples[[parameter]]))
      )
      for(i in seq_along(probs)){
        par_summary <- c(par_summary, stats::quantile(samples[[parameter]], probs = probs[i]))
        names(par_summary)[length(par_summary)] <- probs[i]
      }
      estimates_table <- rbind(estimates_table, par_summary)
      rownames(estimates_table)[nrow(estimates_table)] <- parameter

    }else{
      stop("Uknown parameter type.")
    }
  }

  # prepare output
  estimates_table                    <- data.frame(estimates_table)
  colnames(estimates_table)          <- gsub("X", "", colnames(estimates_table))
  class(estimates_table)             <- c("BayesTools_table", "BayesTools_ensemble_summary", class(estimates_table))
  attr(estimates_table, "type")      <- rep("estimate", ncol(estimates_table))
  attr(estimates_table, "rownames")  <- TRUE
  attr(estimates_table, "title")     <- title
  attr(estimates_table, "footnotes") <- footnotes
  attr(estimates_table, "warnings")  <- warnings

  return(estimates_table)
}

#' @rdname BayesTools_ensemble_tables
ensemble_inference_table <- function(inference, parameters, logBF = FALSE, BF01 = FALSE, title = NULL, footnotes = NULL, warnings = NULL){

  # check input
  check_char(parameters, "parameters", check_length = 0)
  check_list(inference, "inference", check_names = parameters, all_objects = TRUE, allow_other = TRUE)
  check_bool(logBF, "logBF")
  check_bool(BF01,  "BF01")
  check_char(title, "title", allow_NULL = TRUE)
  check_char(footnotes, "footnotes", check_length = 0, allow_NULL = TRUE)
  check_char(warnings, "warnings", check_length = 0, allow_NULL = TRUE)
  if(attr(inference,"conditional"))
    stop("The inference object cannot be 'conditional'.")

  # extract values
  inference_table <- NULL
  n_models        <- NULL
  for(parameter in parameters){
    inference_table <- rbind(inference_table, c(
      "models"     = sum(!attr(inference[[parameter]], "is_null")),
      "prior_prob" = sum(inference[[parameter]][["prior_probs"]][!attr(inference[[parameter]], "is_null")]),
      "post_prob"  = sum(inference[[parameter]][["post_probs"]][!attr(inference[[parameter]], "is_null")] ),
      "BF"         = inference[[parameter]][["BF"]]
    ))
    rownames(inference_table)[nrow(inference_table)] <- parameter
    n_models <- c(n_models, length(attr(inference[[parameter]], "is_null")))
  }
  inference_table <- data.frame(inference_table)

  # format BF
  formatted_BF <- format_BF(inference_table[,"BF"], logBF = logBF, BF01 = BF01)
  inference_table[,"BF"]  <- formatted_BF
  colnames(inference_table)[colnames(inference_table) == "BF"]  <- attr(formatted_BF, "name")

  # prepare output
  class(inference_table)             <- c("BayesTools_table", "BayesTools_ensemble_summary", class(inference_table))
  attr(inference_table, "type")      <- c("n_models", "prior_prob", "post_prob", "inclusion_BF")
  attr(inference_table, "rownames")  <- TRUE
  attr(inference_table, "n_models")  <- n_models
  attr(inference_table, "title")     <- title
  attr(inference_table, "footnotes") <- footnotes
  attr(inference_table, "warnings")  <- warnings

  return(inference_table)
}

#' @rdname BayesTools_ensemble_tables
ensemble_summary_table <- function(models, parameters, title = NULL, footnotes = NULL, warnings = NULL, remove_spike_0 = TRUE, short_name = FALSE){

  # check input
  check_list(models, "models")
  for(i in seq_along(models)){
    model <- models[[i]]
    check_list(model, "model", check_names = c("priors", "inference"), allow_other = TRUE, all_objects = TRUE)
    prior_list <- model[["priors"]]
    check_list(prior_list, "model:priors")
    if(!all(sapply(prior_list, is.prior)))
      stop("'model:priors' must be a list of priors.")
    model_inference <- model[["inference"]]
    check_list(model_inference, "model:inference", check_names = c("m_number", "marglik", "prior_prob", "post_prob", "inclusion_BF"), allow_other = TRUE, all_objects = TRUE)
    check_int(model_inference[["m_number"]],      "model_inference:model_number")
    check_real(model_inference[["marglik"]],      "model_inference:marglik")
    check_real(model_inference[["prior_prob"]],   "model_inference:prior_prob",   lower = 0, upper = 1)
    check_real(model_inference[["post_prob"]],    "model_inference:post_prob",   lower = 0, upper = 1)
    check_real(model_inference[["inclusion_BF"]], "model_inference:inclusion_BF", lower = 0)
  }
  if(is.list(parameters)){
    check_list(parameters, "parameters")
    check_char(names(parameters), "names(parameters)", check_length = FALSE)
    sapply(parameters, check_char, name = "parameters", check_length = FALSE)
  }else{
    check_char(parameters, "parameters", check_length = FALSE)
    if(is.null(names(parameters))){
      names(parameters) <- parameters
    }
  }
  check_char(title, "title", allow_NULL = TRUE)
  check_char(footnotes, "footnotes", check_length = 0, allow_NULL = TRUE)
  check_char(warnings, "warnings", check_length = 0, allow_NULL = TRUE)
  check_bool(short_name, "short_name")


  # create the output
  ensemble_table <- .ensemble_table_foundation(models, parameters, remove_spike_0, short_name)

  ensemble_table <- cbind(
    ensemble_table,
    "prior_prob"   = sapply(models, function(model)model[["inference"]][["prior_prob"]]),
    "marglik"      = sapply(models, function(model)model[["inference"]][["marglik"]]),
    "post_prob"    = sapply(models, function(model)model[["inference"]][["post_prob"]]),
    "BF"           = sapply(models, function(model)model[["inference"]][["inclusion_BF"]])
  )

  # prepare output
  class(ensemble_table)             <- c("BayesTools_table", "BayesTools_ensemble_summary", class(ensemble_table))
  attr(ensemble_table, "type")      <- c("integer", rep("prior", length(parameters)), "prior_prob", "marglik", "post_prob", "inclusion_BF")
  attr(ensemble_table, "rownames")  <- FALSE
  attr(ensemble_table, "title")     <- title
  attr(ensemble_table, "footnotes") <- footnotes
  attr(ensemble_table, "warnings")  <- warnings

  return(ensemble_table)
}

#' @rdname BayesTools_ensemble_tables
ensemble_diagnostics_table <- function(models, parameters, title = NULL, footnotes = NULL, warnings = NULL, remove_spike_0 = TRUE, short_name = FALSE){

  # check input
  check_list(models, "models")
  for(i in seq_along(models)){
    model <- models[[i]]
    check_list(model, "model", check_names = c("priors", "fit_summary", "inference"), allow_other = TRUE, all_objects = TRUE)
    prior_list <- model[["priors"]]
    check_list(prior_list, "model:priors")
    if(!all(sapply(prior_list, is.prior)))
      stop("'model:priors' must be a list of priors.")
    if(!is.null(model[["fit_summary"]]) && !inherits(model[["fit_summary"]], "BayesTools_runjags_summary"))
      stop("'fit_summary' must be a runjags summary generated by 'runjags_estimates_table()'.")
    model_inference <- model[["inference"]]
    check_list(model_inference, "model:inference", check_names = "m_number", allow_other = TRUE, all_objects = TRUE)
    check_int(model_inference[["m_number"]], "model_inference:model_number")
  }
  if(is.list(parameters)){
    check_list(parameters, "parameters")
    check_char(names(parameters), "names(parameters)", check_length = FALSE)
    sapply(parameters, check_char, name = "parameters", check_length = FALSE)
  }else{
    check_char(parameters, "parameters", check_length = FALSE)
    if(is.null(names(parameters))){
      names(parameters) <- parameters
    }
  }
  check_char(title, "title", allow_NULL = TRUE)
  check_char(footnotes, "footnotes", check_length = 0, allow_NULL = TRUE)
  check_char(warnings, "warnings", check_length = 0, allow_NULL = TRUE)
  check_bool(short_name, "short_name")

  # create the output
  ensemble_table <- .ensemble_table_foundation(models, parameters, remove_spike_0, short_name)

  ensemble_table <- cbind(
    ensemble_table,
    "max_MCMC_error"    = sapply(models, function(model){
      MCMC_error <- model[["fit_summary"]][,"MCMC_error"]
      if(all(is.na(MCMC_error))){
        return(NA)
      }else{
        max(MCMC_error,    na.rm = TRUE)
      }
    }),
    "max_MCMC_SD_error" = sapply(models, function(model){
      MCMC_SD_error <- model[["fit_summary"]][,"MCMC_SD_error"]
      if(all(is.na(MCMC_SD_error))){
        return(NA)
      }else{
        max(MCMC_SD_error, na.rm = TRUE)
      }
    }),
    "min_ESS"     = sapply(models, function(model){
      ESS <- model[["fit_summary"]][,"ESS"]
      if(all(is.na(ESS))){
        return(NA)
      }else{
        min(ESS, na.rm = TRUE)
      }
    }),
    "max_R_hat"   = sapply(models, function(model){
      Rhat <- model[["fit_summary"]][,"R_hat"]
      if(all(is.na(Rhat))){
        return(NA)
      }else{
        max(Rhat, na.rm = TRUE)
      }
    })
  )

  # prepare output
  class(ensemble_table)             <- c("BayesTools_table", "BayesTools_ensemble_summary", class(ensemble_table))
  attr(ensemble_table, "type")      <- c("integer", rep("prior", length(parameters)), "max_MCMC_error", "max_MCMC_SD_error", "min_ESS", "max_R_hat")
  attr(ensemble_table, "rownames")  <- FALSE
  attr(ensemble_table, "title")     <- title
  attr(ensemble_table, "footnotes") <- footnotes
  attr(ensemble_table, "warnings")  <- warnings

  return(ensemble_table)
}

.ensemble_table_foundation <- function(models, parameters, remove_spike_0, short_name){

  model_rows <- list()
  for(i in seq_along(models)){

    model_row <- list()
    model_row[["Model"]] <- models[[i]][["inference"]][["m_number"]]

    for(p in seq_along(parameters)){
      if(is.list(parameters)){
        if(sum(names(models[[i]][["priors"]]) %in% parameters[[p]]) == 1){
          temp_prior <- models[[i]][["priors"]][[parameters[[p]][parameters[[p]] %in% names(models[[i]][["priors"]])]]]
        }else if(sum(names(models[[i]][["priors"]]) %in% parameters[[p]]) == 0){
          temp_prior <- prior_none()
        }else{
          stop("More than one prior matching the specified grouping.")
        }
      }else{
        if(any(names(models[[i]][["priors"]]) == parameters[p])){
          temp_prior <- models[[i]][["priors"]][[parameters[p]]]
        }else{
          temp_prior <- prior_none()
        }
      }
      if(remove_spike_0 && is.prior.point(temp_prior) && temp_prior[["parameters"]][["location"]] == 0){
        model_row[[names(parameters)[p]]] <- ""
      }else if(is.prior.none(temp_prior)){
        model_row[[names(parameters)[p]]] <- ""
      }else{
        model_row[[names(parameters)[p]]] <- print(temp_prior, silent = TRUE, short_name = short_name)
      }
    }
    model_rows[[i]] <- model_row
  }
  summary_table <- data.frame(do.call(rbind, model_rows))
  for(i in 1:ncol(summary_table)){
    summary_table[,i] <- unlist(summary_table[,i])
  }
  colnames(summary_table) <- c("Model", names(parameters))

  return(summary_table)
}


#' @title Create BayesTools model tables
#'
#' @description Creates model summary based on a model objects or
#' provides estimates table for a runjags fit.
#'
#' @param model model object containing a list of \code{priors}
#' and \code{inference} object, The \code{inference} must be a
#' named list with information about the model: model number
#' \code{m_number}, marginal likelihood \code{marglik}, prior and
#' posterior probability \code{prior_prob} and \code{post_prob},
#' and model inclusion Bayes factor \code{inclusion_BF}
#' @param fit runjags model fit
#' @param prior_list list of prior distributions
#' @param transformations named list of transformations to be applied
#' to specific parameters
#' @param model_description named list with additional description
#' to be added to the table
#' @inheritParams BayesTools_ensemble_tables
#'
#'
#' @return \code{model_summary_table} returns a table with
#' overview of the fitted model, \code{runjags_estimates_table} returns
#' a table with MCMC estimates, and \code{runjags_estimates_empty_table}
#' returns an empty estimates table. All of the tables are objects of
#' class 'BayesTools_table'.
#'
#' @export model_summary_table
#' @export runjags_estimates_table
#' @export runjags_estimates_empty_table
#' @name BayesTools_model_tables
#'
#' @seealso [BayesTools_ensemble_tables]
NULL

#' @rdname BayesTools_model_tables
model_summary_table <- function(model, model_description = NULL, title = NULL, footnotes = NULL, warnings = NULL, remove_spike_0 = TRUE, short_name = FALSE){

  # check input
  check_list(model, "model", check_names = c("priors", "inference"), allow_other = TRUE, all_objects = TRUE)
  prior_list <- model[["priors"]]
  check_list(prior_list, "model:priors")
  if(!all(sapply(prior_list, is.prior)))
    stop("'model:priors' must be a list of priors.")
  model_inference <- model[["inference"]]
  check_list(model_inference, "model:inference", check_names = c("m_number", "marglik", "prior_prob", "post_prob", "inclusion_BF"), allow_other = TRUE, all_objects = TRUE)
  check_int(model_inference[["m_number"]],      "model_inference:model_number")
  check_real(model_inference[["marglik"]],      "model_inference:marglik")
  check_real(model_inference[["prior_prob"]],   "model_inference:prior_prob",   lower = 0, upper = 1)
  check_real(model_inference[["post_prob"]],    "model_inference:post_prob",   lower = 0, upper = 1)
  check_real(model_inference[["inclusion_BF"]], "model_inference:inclusion_BF", lower = 0)
  check_list(model_description, "model_description", allow_NULL = TRUE)
  check_bool(short_name, "short_name")
  check_char(title, "title", allow_NULL = TRUE)
  check_char(footnotes, "footnotes", check_length = 0, allow_NULL = TRUE)
  check_char(warnings, "warnings", check_length = 0, allow_NULL = TRUE)

  # prepare the columns
  summary_names  <- c(
    "Model",
    if(!is.null(model_description)) names(model_description),
    "Prior prob.",
    "log(marglik)",
    "Post. prob.",
    "Inclusion BF")
  summary_values <- c(
    model_inference[["m_number"]],
    if(!is.null(model_description)) unlist(model_description),
    .format_column(model_inference[["prior_prob"]],   "probability"),
    .format_column(model_inference[["marglik"]],      "marglik"),
    .format_column(model_inference[["post_prob"]],    "probability"),
    .format_column(model_inference[["inclusion_BF"]], "BF"))
  summary_priors <- "Parameter prior distributions"
  for(i in seq_along(prior_list)){
    if(remove_spike_0 && is.prior.point(prior_list[[i]]) && prior_list[[i]][["parameters"]][["location"]] == 0){
      next
    }else if(is.prior.weightfunction(prior_list[[i]]) | is.prior.PET(prior_list[[i]]) | is.prior.PEESE(prior_list[[i]])){
      summary_priors <- c(summary_priors, print(prior_list[[i]], silent = TRUE, short_name = short_name))
    }else if(is.prior.simple(prior_list[[i]])){
      summary_priors <- c(summary_priors, paste0(names(prior_list)[i], " ~ " , print(prior_list[[i]], silent = TRUE, short_name = short_name)))
    }else if(is.prior.point(prior_list[[i]])){
      summary_priors <- c(summary_priors, paste0(names(prior_list)[i], " = " , print(prior_list[[i]], silent = TRUE, short_name = short_name)))
    }
  }

  if(length(summary_names) > length(summary_priors)){
    summary_priors <- c(summary_priors, rep("", length(summary_names) - length(summary_priors)))
  }else if(length(summary_names) < length(summary_priors)){
    summary_names  <- c(summary_names,  rep("", length(summary_priors) - length(summary_names)))
    summary_values <- c(summary_values, rep("", length(summary_priors) - length(summary_values)))
  }
  summary_names <- paste0(summary_names, "  ")

  summary_table <- data.frame(cbind(
    summary_names,
    summary_values,
    rep("           ", length(summary_names)),
    summary_priors
  ))
  names(summary_table) <- NULL

  # prepare output
  class(summary_table)             <- c("BayesTools_table", class(summary_table))
  attr(summary_table, "type")      <- c("string_left", "string", "string", "prior")
  attr(summary_table, "rownames")  <- FALSE
  attr(summary_table, "as.matrix") <- TRUE
  attr(summary_table, "title")     <- title
  attr(summary_table, "footnotes") <- footnotes
  attr(summary_table, "warnings")  <- warnings

  return(summary_table)
}

#' @rdname BayesTools_model_tables
runjags_estimates_table  <- function(fit, prior_list, transformations = NULL, title = NULL, footnotes = NULL, warnings = NULL, remove_spike_0 = TRUE){

  # check fits
  if(!inherits(fit, "runjags"))
    stop("'fit' must be a runjags fit")
  check_list(prior_list, "prior_list")
  if(!all(sapply(prior_list, is.prior)))
    stop("'prior_list' must be a list of priors.")
  check_list(transformations, "transformations", allow_NULL = TRUE)
  if(!is.null(transformations) && any(!sapply(transformations, function(trans)is.function(trans[["fun"]]))))
    stop("'transformations' must be list of functions in the 'fun' element.")
  check_char(title, "title", allow_NULL = TRUE)
  check_char(footnotes, "footnotes", check_length = 0, allow_NULL = TRUE)
  check_char(warnings, "warnings", check_length = 0, allow_NULL = TRUE)
  check_bool(remove_spike_0, "remove_spike_0")

  # obtain model information
  invisible(utils::capture.output(runjags_summary <- suppressWarnings(summary(fit, silent.jags = TRUE))))
  runjags_summary <- data.frame(runjags_summary)
  model_samples   <- suppressWarnings(coda::as.mcmc(fit))

  # deal with missing median in case of non-stochastic variables
  if(!any(colnames(runjags_summary) == "Median")){
    runjags_summary[,"Median"] <- NA
  }

  # apply transformations
  if(!is.null(transformations)){
    for(par in names(transformations)){
      model_samples[,par] <- do.call(transformations[[par]][["fun"]], c(list(model_samples[,par]), transformations[[par]][["arg"]]))
      runjags_summary[par, "Mean"]    <- do.call(transformations[[par]][["fun"]], c(list(runjags_summary[par, "Mean"]), transformations[[par]][["arg"]]))
      runjags_summary[par, "SD"]      <- sd(model_samples[,par])
      runjags_summary[par, "Median"]  <- do.call(transformations[[par]][["fun"]], c(list(runjags_summary[par, "Median"]), transformations[[par]][["arg"]]))
      runjags_summary[par, "MCerr"]   <- do.call(transformations[[par]][["fun"]], c(list(runjags_summary[par, "MCerr"]), transformations[[par]][["arg"]]))
      runjags_summary[par, "MC.ofSD"] <- runjags_summary[par, "MCerr"] / runjags_summary[par, "SD"]
    }
  }

  # change HPD to quantile intervals
  for(par in rownames(runjags_summary)){
    runjags_summary[par, "Lower95"] <- stats::quantile(model_samples[,par], .025)
    runjags_summary[par, "Upper95"] <- stats::quantile(model_samples[,par], .975)
  }

  # remove un-wanted columns
  runjags_summary <- runjags_summary[,!colnames(runjags_summary) %in% c("Mode", "AC.10"),drop = FALSE]

  # remove un-wanted estimates (or support values)
  for(i in seq_along(prior_list)){
    if(is.prior.weightfunction(prior_list[[i]])){
      # remove etas
      if(prior_list[[i]][["distribution"]] %in% c("one.sided", "two.sided")){
        runjags_summary <- runjags_summary[!grepl("eta", rownames(runjags_summary)),,drop=FALSE]
      }
      # remove wrong diagnostics for the constant
      runjags_summary[max(grep("omega", rownames(runjags_summary))),c("MCerr", "MC.ofSD","SSeff","psfr")] <- NA
      # reorder
      runjags_summary[grep("omega", rownames(runjags_summary)),] <- runjags_summary[rev(grep("omega", rownames(runjags_summary))),]
      # rename
      omega_cuts    <- weightfunctions_mapping(prior_list[i], cuts_only = TRUE)
      omega_names   <- sapply(1:(length(omega_cuts)-1), function(i)paste0("omega[",omega_cuts[i],",",omega_cuts[i+1],"]"))
      rownames(runjags_summary)[grep("omega", rownames(runjags_summary))] <- omega_names
    }else if(remove_spike_0 && is.prior.point(prior_list[[i]]) && prior_list[[i]][["parameters"]][["location"]] == 0){
      runjags_summary <- runjags_summary[rownames(runjags_summary) != names(prior_list)[i],,drop=FALSE]
    }else if(is.prior.simple(prior_list[[i]]) &&  prior_list[[i]][["distribution"]] == "invgamma"){
      runjags_summary <- runjags_summary[rownames(runjags_summary) != paste0("inv_",names(prior_list)[i]),,drop=FALSE]
    }
  }

  # rename the rest
  colnames(runjags_summary)[colnames(runjags_summary) == "Lower95"] <- "lCI"
  colnames(runjags_summary)[colnames(runjags_summary) == "Upper95"] <- "uCI"
  colnames(runjags_summary)[colnames(runjags_summary) == "MCerr"]   <- "MCMC_error"
  colnames(runjags_summary)[colnames(runjags_summary) == "MC.ofSD"] <- "MCMC_SD_error"
  colnames(runjags_summary)[colnames(runjags_summary) == "SSeff"]   <- "ESS"
  colnames(runjags_summary)[colnames(runjags_summary) == "psrf"]    <- "R_hat"

  # change the SD error to a fraction
  runjags_summary[, "MCMC_SD_error"] <- runjags_summary[, "MCMC_SD_error"] / 100

  # reorder the columns
  runjags_summary <- runjags_summary[,c("Mean", "SD", "lCI", "Median", "uCI", "MCMC_error", "MCMC_SD_error", "ESS", "R_hat"), drop = FALSE]

  # prepare output
  class(runjags_summary)             <- c("BayesTools_table", "BayesTools_runjags_summary", class(runjags_summary))
  attr(runjags_summary, "type")      <- c(rep("estimate", 5), "MCMC_error", "MCMC_SD_error", "ESS", "R_hat")
  attr(runjags_summary, "rownames")  <- TRUE
  attr(runjags_summary, "title")     <- title
  attr(runjags_summary, "footnotes") <- footnotes
  attr(runjags_summary, "warnings")  <- warnings

  return(runjags_summary)
}

#' @rdname BayesTools_model_tables
runjags_estimates_empty_table <- function(title = NULL, footnotes = NULL, warnings = NULL){

  empty_table <- data.frame(matrix(nrow = 0, ncol = 9))
  colnames(empty_table) <- c("Mean", "SD", "lCI", "Median", "uCI", "MCMC_error", "MCMC_SD_error", "ESS", "R_hat")

  class(empty_table)             <- c("BayesTools_table", "BayesTools_runjags_summary", class(empty_table))
  attr(empty_table, "type")      <- c(rep("estimate", 5), "MCMC_error", "MCMC_SD_error", "ESS", "R_hat")
  attr(empty_table, "rownames")  <- FALSE
  attr(empty_table, "title")     <- title
  attr(empty_table, "footnotes") <- footnotes
  attr(empty_table, "warnings")  <- warnings

  return(empty_table)
}


#' @title Print a BayesTools table
#'
#' @param x a BayesTools_values_tables
#' @param ... additional arguments.
#'
#' @return \code{print.BayesTools_table} returns \code{NULL}.
#'
#' @exportS3Method
print.BayesTools_table <- function(x, ...){

  # print formatting
  for(i in seq_along(attr(x, "type"))){
    x[,i]          <- .format_column(x[,i], attr(x, "type")[i], attr(x, "n_models")[i])
    colnames(x)[i] <- .format_column_names(colnames(x)[i], attr(x, "type")[i], x[,i])
  }

  # print title
  if(!is.null(attr(x, "title"))){
    cat(paste0(attr(x, "title"), "\n"))
  }

  # print the table
  print(as.data.frame(x), quote = FALSE, right = TRUE, row.names = attr(x, "rownames"))

  # print footnotes
  for(i in seq_along(attr(x, "footnotes"))){
    cat(paste0(attr(x, "footnotes")[i], "\n"))
  }

  # print warnings in red
  for(i in seq_along(attr(x, "warnings"))){
    cat(paste0("\033[0;31m", attr(x, "warnings")[i], "\033[0m\n"))
  }

  return(invisible())
}

#' @title Format Bayes factor
#'
#' @description Formats Bayes factor
#'
#' @param BF Bayes factor(s)
#' @param logBF log(BF)
#' @param BF01 1/BF
#'
#' @return \code{format_BF} returns a formatted Bayes factor.
#'
#' @export
format_BF <- function(BF, logBF = FALSE, BF01 = FALSE){

  check_real(BF, "BF", lower = 0, check_length = FALSE)
  check_bool(logBF, "logBF")
  check_bool(BF01,  "BF01")

  name <- "BF"

  if(BF01){
    BF   <- 1/BF
    name <- "1/BF"
  }
  if(logBF){
    BF   <- log(BF)
    name <- paste0("log(", name, ")")
  }

  attr(BF, "name")  <- name
  attr(BF, "logBF") <- logBF
  attr(BF, "BF01")  <- BF01

  return(BF)
}


.format_column       <- function(x, type, n_models){
  if(is.null(x)){
    return(x)
  }else{
    return(switch(
      type,
      "integer"         = round(x),
      "prior"           = .center_priors(x),
      "string_left"     = .string_left(x),
      "string"          = x,
      "estimate"        = format(round(x, digits = 3), nsmall = 3),
      "prior_prob"      = format(round(x, digits = 3), nsmall = 3),
      "post_prob"       = format(round(x, digits = 3), nsmall = 3),
      "probability"     = format(round(x, digits = 3), nsmall = 3),
      "marglik"         = format(round(x, digits = 2), nsmall = 2),
      "BF"              = format(round(x, digits = 3), nsmall = 3),
      "inclusion_BF"    = format(round(x, digits = 3), nsmall = 3),
      "n_models"        = paste0(round(x), "/", n_models),
      "ESS"             = round(x),
      "R_hat"           = format(round(x, digits = 3), nsmall = 3),
      "MCMC_error"      = format(round(x, digits = 5), nsmall = 5),
      "MCMC_SD_error"   = format(round(x, digits = 3), nsmall = 3),
      "min_ESS"             = round(x),
      "max_R_hat"           = format(round(x, digits = 3), nsmall = 3),
      "max_MCMC_error"      = format(round(x, digits = 5), nsmall = 5),
      "max_MCMC_SD_error"   = format(round(x, digits = 3), nsmall = 3)
    ))
  }
}
.format_column_names <- function(x, type, values){
  if(is.null(x)){
    return(x)
  }else{
    return(switch(
      type,
      "integer"         = x,
      "prior"           = .string_center(paste0("Prior ", x), values),
      "string_left"     = .string_left(x, values),
      "string"          = x,
      "estimate"        = x,
      "probability"     = x,
      "prior_prob"      = "Prior prob.",
      "post_prob"       = "Post. prob.",
      "marglik"         = "log(marglik)",
      "BF"              = x,
      "inclusion_BF"    = paste0("Inclusion ", x),
      "n_models"        = "Models",
      "ESS"             = "ESS",
      "R_hat"           = "R-hat",
      "MCMC_error"      = "error(MCMC)",
      "MCMC_SD_error"   = "SD/error(MCMC)",
      "min_ESS"             = "min(ESS)",
      "max_R_hat"           = "max(R-hat)",
      "max_MCMC_error"      = "max[error(MCMC)]",
      "max_MCMC_SD_error"   = "max[SD/error(MCMC)]",
    ))
  }
}
.center_priors <- function(x){

  if(any(grepl("~", x) | grepl("=", x))){

    position_tilda  <- regexpr("~", x)
    position_equal  <- regexpr("=", x)
    from_right      <- sapply(seq_along(x), function(i){
      if(position_tilda[i] != -1){
        return(nchar(x[i]) - position_tilda[i])
      }else if(position_equal[i] != -1){
        return(nchar(x[i]) - position_equal[i])
      }else{
        return(0)
      }
    })
    add_to_right  <- ifelse(from_right == 0, 0, max(from_right) - from_right)
    x <- paste0(x, sapply(seq_along(x), function(i)paste0(rep(" ", add_to_right[i]), collapse = "")))

  }

  return(x)
}
.string_left   <- function(x, reference = x){

  if(length(x) > 0){

    add_to_right <- max(nchar(reference)) - nchar(x)
    x <- paste0(x, sapply(seq_along(x), function(i)paste0(rep(" ", add_to_right[i]), collapse = "")))

  }

  return(x)
}
.string_center <- function(x, reference = x){

  if(length(x) > 0){

    add_to_sides <- max(nchar(reference)) - nchar(x)
    add_to_sides <- ifelse(add_to_sides < 0, 0, add_to_sides)
    x <- paste0( sapply(seq_along(x), function(i)paste0(rep(" ", round(add_to_sides[i]/2)), collapse = "")), x, sapply(seq_along(x), function(i)paste0(rep(" ", round(add_to_sides[i]/2)), collapse = "")))

  }

  return(x)
}
