% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/singlepaths.R
\name{singlepaths}
\alias{singlepaths}
\title{Plot parameter paths for BTLLasso}
\usage{
singlepaths(model, rescale = FALSE, colors = NULL, equal.ranges = FALSE,
  plot.X = TRUE, plot.Z1 = TRUE, plot.Z2 = TRUE, plot.intercepts = TRUE,
  plot.order.effects = TRUE, x.axis = c("loglambda", "lambda", "penalty"),
  columns = NULL, subs.X = NULL, subs.Z1 = NULL, subs.Z2 = NULL)
}
\arguments{
\item{model}{BTLLasso or cv.BTLLasso object}

\item{rescale}{Should the parameter estimates be rescaled for plotting? Only 
applies if \code{scale = TRUE} was specified in \code{BTLLasso} or \code{cv.BTLLasso}.}

\item{colors}{Optional. If specified, vector with length equal to the number
of objects. Each object can be represented by another color.}

\item{equal.ranges}{Should all single plots (for different covariates) have
equal ranges on the y-axes. FALSE by default.}

\item{plot.X}{Should paths for variables in \code{X} (if present) be plotted?}

\item{plot.Z1}{Should paths for variables in \code{Z1} (if present) be plotted?}

\item{plot.Z2}{Should paths for variables in \code{Z2} (if present) be plotted?}

\item{plot.intercepts}{Should paths for intercepts be plotted separately?}

\item{plot.order.effects}{Should paths for order effects be plotted separately?}

\item{x.axis}{Should the paths be plotted against log(lambda+1), against lambda or
against the scaled penalty term (between 0 and 1)?}

\item{columns}{How many columns should be used to arrange the plots. If unspecified, plots 
are arranged automatically in a quadratic manner.}

\item{subs.X}{Optional vector of subtitles for variables in \code{X}. Can be used
to note the encoding of the single covariates, especially for dummy
variables.}

\item{subs.Z1}{Optional vector of subtitles for variables in \code{Z1}. Can be used
to note the encoding of the single covariates, especially for dummy
variables.}

\item{subs.Z2}{Optional vector of subtitles for variables in \code{Z2}. Can be used
to note the encoding of the single covariates, especially for dummy
variables.}
}
\description{
Plots single paths for every parameter of a BTLLasso object or a cv.BTLLasso
object. In contrast, to \code{\link{paths}}, one plot per covariate is
created, every single parameter is illustrated by one path. For cv.BTLLasso
object, the optimal model according to the cross-validation is marked by a
vertical dashed line.
}
\details{
Plots for BTLLasso and cv.BTLLasso objects only differ by the additional
vertical line indicating the optimal model according to cross-validation.
}
\examples{

\dontrun{
##### Example with simulated data set containing X, Z1 and Z2
data(SimData)

## Specify tuning parameters
lambda <- exp(seq(log(151), log(1.05), length = 30)) - 1

## Specify control argument
## -> allow for object-specific order effects and penalize intercepts
ctrl <- ctrl.BTLLasso(penalize.intercepts = TRUE, object.order.effect = TRUE,
                      penalize.order.effect.diffs = TRUE)

## Simple BTLLasso model for tuning parameters lambda
m.sim <- BTLLasso(Y = SimData$Y, X = SimData$X, Z1 = SimData$Z1, 
                  Z2 = SimData$Z2, lambda = lambda, control = ctrl)
print(m.sim)

singlepaths(m.sim)

## Cross-validate BTLLasso model for tuning parameters lambda
set.seed(5)
m.sim.cv <- cv.BTLLasso(Y = SimData$Y, X = SimData$X, Z1 = SimData$Z1, 
                        Z2 = SimData$Z2, lambda = lambda, control = ctrl)
print(m.sim.cv)

singlepaths(m.sim.cv, plot.order.effect = FALSE, 
            plot.intercepts = FALSE, plot.Z2 = FALSE)
paths(m.sim.cv, y.axis = 'L2')

## Example for bootstrap confidence intervals for illustration only
## Don't calculate bootstrap confidence intervals with B = 10!!!!
set.seed(5)
m.sim.boot <- boot.BTLLasso(m.sim.cv, B = 10, cores = 10)
print(m.sim.boot)
ci.BTLLasso(m.sim.boot)


##### Example with small version from GLES data set
data(GLESsmall)

## extract data and center covariates for better interpretability
Y <- GLESsmall$Y
X <- scale(GLESsmall$X, scale = FALSE)
Z1 <- scale(GLESsmall$Z1, scale = FALSE)

## vector of subtitles, containing the coding of the X covariates
subs.X <- c('', 'female (1); male (0)')

## vector of tuning parameters
lambda <- exp(seq(log(61), log(1), length = 30)) - 1


## compute BTLLasso model 
m.gles <- BTLLasso(Y = Y, X = X, Z1 = Z1, lambda = lambda)
print(m.gles)

singlepaths(m.gles, subs.X = subs.X)
paths(m.gles, y.axis = 'L2')

## Cross-validate BTLLasso model 
m.gles.cv <- cv.BTLLasso(Y = Y, X = X, Z1 = Z1, lambda = lambda)
print(m.gles.cv)

singlepaths(m.gles.cv, subs.X = subs.X)
}

}
\author{
Gunther Schauberger\cr \email{gunther@stat.uni-muenchen.de}\cr
\url{http://www.statistik.lmu.de/~schauberger/}
}
\references{
Schauberger, Gunther and Tutz, Gerhard (2015): Modelling
Heterogeneity in Paired Comparison Data - an L1 Penalty Approach with an
Application to Party Preference Data, \emph{Department of Statistics, LMU
Munich}, Technical Report 183

Schauberger, Gunther, Groll Andreas and Tutz, Gerhard (2016): Modelling 
Football Results in the German Bundesliga Using Match-specific Covariates, 
\emph{Department of Statistics, LMU Munich}, Technical Report 197
}
\seealso{
\code{\link{BTLLasso}}, \code{\link{cv.BTLLasso}},
\code{\link{paths}}
}
\keyword{BTLLasso}
\keyword{parameter}
\keyword{paths}

