% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/BTLLasso.R
\name{ctrl.BTLLasso}
\alias{ctrl.BTLLasso}
\title{Control function for BTLLasso}
\usage{
ctrl.BTLLasso(adaptive = TRUE, scale = TRUE, norm = c("L1", "L2"),
  epsilon = 1e-04, lambda2 = 1e-04, c = 1e-09, precision = 3,
  weight.penalties = TRUE, include.intercepts = TRUE,
  order.effect = FALSE, object.order.effect = FALSE, order.center = FALSE,
  name.order = "Order", penalize.intercepts = FALSE, penalize.X = TRUE,
  penalize.Z2 = FALSE, penalize.Z1.absolute = TRUE,
  penalize.Z1.diffs = TRUE, penalize.order.effect.absolute = TRUE,
  penalize.order.effect.diffs = FALSE)
}
\arguments{
\item{adaptive}{Should adaptive lasso be used? Default is TRUE.}

\item{scale}{Should the covariates be scaled so that they are on comparable scales? Default is TRUE.
Variables will be properly scaled AND centered. Please note that results will refer to scaled covariates.
If \code{adaptive = TRUE} scaling is not necessary to keep penalties comparable.}

\item{norm}{Specifies the norm used in the penalty term. Currently, only
'L1' and 'L2' are possible. Default is to 'L1', only 'L1' allows for
clustering and variable selection.}

\item{epsilon}{Threshold value for convergence of the algorithm.}

\item{lambda2}{Tuning parameter for ridge penalty on all coefficients.
Should be small, only used to stabilize results.}

\item{c}{Internal parameter for the quadratic approximation of the L1
penalty. Should be sufficiently small. For details see
\code{\link[gvcm.cat]{cat_control}}.}

\item{precision}{Precision for final parameter estimates, specifies number of decimals.}

\item{weight.penalties}{Should the penalties across the different model components 
(i.e. intercepts, order effects, X, Z1, Z2) be weighted according to the number of
penalties included? Default is \code{TRUE} to minimize the risk of selection bias
across different model components.}

\item{include.intercepts}{Should intercepts be included in the model?}

\item{order.effect}{Should a global order effect (corresponding to home effect
in sports applications) be included in the model?}

\item{object.order.effect}{Should object-specific order effects (corresponding to home effects
in sports applications) be included in the model?}

\item{order.center}{Should (in case of object-specific order effects) the order effects be centered
in the design matrix? Centering is equivalent to the coding scheme of effect coding instead of 
dummy coding.}

\item{name.order}{How should the order effect(s) be called in plots or prints.}

\item{penalize.intercepts}{Should intercepts be penalized? If \code{TRUE},
all pairwise differences between intercepts are penalized.}

\item{penalize.X}{Should effects from X matrix be penalized? If \code{TRUE},
all pairwise differences corresponding to one covariate are penalized.}

\item{penalize.Z2}{Should absolute values of effects from Z2 matrix be
penalized?}

\item{penalize.Z1.absolute}{Should absolute values of effects from Z1 matrix
be penalized?}

\item{penalize.Z1.diffs}{Should differences of effects from Z1 matrix be
penalized? If \code{TRUE}, all pairwise differences corresponding to one
covariate are penalized.}

\item{penalize.order.effect.absolute}{Should absolute values of order effect(s) be penalized?
Only relevant if either \code{object.order.effect = TRUE} or \code{order.effect = TRUE}.}

\item{penalize.order.effect.diffs}{Should differences of order effects be
penalized? If \code{TRUE}, all pairwise differences are penalized. Only relevant if 
\code{object.order.effect = TRUE}}
}
\description{
Control parameters for different penalty terms and for tuning the fitting algorithm.
}
\examples{

\dontrun{
##### Example with simulated data set containing X, Z1 and Z2
data(SimData)

## Specify tuning parameters
lambda <- exp(seq(log(151), log(1.05), length = 30)) - 1

## Specify control argument
## -> allow for object-specific order effects and penalize intercepts
ctrl <- ctrl.BTLLasso(penalize.intercepts = TRUE, object.order.effect = TRUE,
                      penalize.order.effect.diffs = TRUE)

## Simple BTLLasso model for tuning parameters lambda
m.sim <- BTLLasso(Y = SimData$Y, X = SimData$X, Z1 = SimData$Z1, 
                  Z2 = SimData$Z2, lambda = lambda, control = ctrl)
print(m.sim)

singlepaths(m.sim)

## Cross-validate BTLLasso model for tuning parameters lambda
set.seed(5)
m.sim.cv <- cv.BTLLasso(Y = SimData$Y, X = SimData$X, Z1 = SimData$Z1, 
                        Z2 = SimData$Z2, lambda = lambda, control = ctrl)
print(m.sim.cv)

singlepaths(m.sim.cv, plot.order.effect = FALSE, 
            plot.intercepts = FALSE, plot.Z2 = FALSE)
paths(m.sim.cv, y.axis = 'L2')

## Example for bootstrap confidence intervals for illustration only
## Don't calculate bootstrap confidence intervals with B = 10!!!!
set.seed(5)
m.sim.boot <- boot.BTLLasso(m.sim.cv, B = 10, cores = 10)
print(m.sim.boot)
ci.BTLLasso(m.sim.boot)


##### Example with small version from GLES data set
data(GLESsmall)

## extract data and center covariates for better interpretability
Y <- GLESsmall$Y
X <- scale(GLESsmall$X, scale = FALSE)
Z1 <- scale(GLESsmall$Z1, scale = FALSE)

## vector of subtitles, containing the coding of the X covariates
subs.X <- c('', 'female (1); male (0)')

## vector of tuning parameters
lambda <- exp(seq(log(61), log(1), length = 30)) - 1


## compute BTLLasso model 
m.gles <- BTLLasso(Y = Y, X = X, Z1 = Z1, lambda = lambda)
print(m.gles)

singlepaths(m.gles, subs.X = subs.X)
paths(m.gles, y.axis = 'L2')

## Cross-validate BTLLasso model 
m.gles.cv <- cv.BTLLasso(Y = Y, X = X, Z1 = Z1, lambda = lambda)
print(m.gles.cv)

singlepaths(m.gles.cv, subs.X = subs.X)
}
}
\author{
Gunther Schauberger\cr \email{gunther@stat.uni-muenchen.de}\cr
\url{http://www.statistik.lmu.de/~schauberger/}
}
\references{
Schauberger, Gunther and Tutz, Gerhard (2015): Modelling
Heterogeneity in Paired Comparison Data - an L1 Penalty Approach with an
Application to Party Preference Data, \emph{Department of Statistics, LMU
Munich}, Technical Report 183

Schauberger, Gunther, Groll Andreas and Tutz, Gerhard (2016): Modelling 
Football Results in the German Bundesliga Using Match-specific Covariates, 
\emph{Department of Statistics, LMU Munich}, Technical Report 197
}
\seealso{
\code{\link{BTLLasso}}, \code{\link{cv.BTLLasso}}
}
\keyword{BTLLasso}
\keyword{control}

