% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/BTLLasso.R
\name{BTLLasso}
\alias{BTLLasso}
\title{Function to perform BTLLasso}
\usage{
BTLLasso(Y, X = NULL, Z1 = NULL, Z2 = NULL, lambda,
  control = ctrl.BTLLasso(), trace = TRUE)
}
\arguments{
\item{Y}{A \code{response.BTLLasso} object created by
\code{\link{response.BTLLasso}}.}

\item{X}{Matrix containing all \bold{subject-specific covariates} that are
to be included with \bold{object-specific effects}. One row represents one
subject, one column represents one covariate. X has to be standardized.}

\item{Z1}{Matrix containing all \bold{object-subject-specific covariates}
that are to be included with \bold{object-specific effects}. One row
represents one subject, one column represents one combination between
covariate and object. Column names have to follow the scheme
'firstvar.object1',...,'firstvar.objectm',...,'lastvar.objectm'. The object
names 'object1',...,'objectm' have to be identical to the object names used
in the \code{response.BTLLasso} object \code{Y}. The variable names and the
object names have to be separated by '.'.  The rownames of the matrix",
Z.name, "have to be equal to the subjects specified in the response object.
Z1 has to be standardized.}

\item{Z2}{Matrix containing all \bold{object-subject-specific covariates or
object-specific covariates} that are to be included with \bold{global
effects}. One row represents one subject, one column represents one
combination between covariate and object. Column names have to follow the
scheme 'firstvar.object1',...,'firstvar.objectm',...,'lastvar.objectm'. The
object names 'object1',...,'objectm' have to be identical to the object
names used in the \code{response.BTLLasso} object \code{Y}. The variable
names and the object names have to be separated by '.'.  The rownames of the
matrix", Z.name, "have to be equal to the subjects specified in the response
object. Z2 has to be standardized.}

\item{lambda}{Vector of tuning parameters.}

\item{control}{Function for control arguments, mostly for internal use. See
also \code{\link{ctrl.BTLLasso}}.}

\item{trace}{Should the trace of the BTLLasso algorithm be printed?}
}
\value{
\item{coefs}{Matrix containing all (original) coefficients, one row
per tuning parameter, one column per coefficient.} 
\item{coefs.repar}{Matrix
containing all reparameterized (for symmetric side constraint) coefficients,
one row per tuning parameter, one column per coefficient.}
\item{logLik}{Vector of log-likelihoods, one value per tuning parameter.}
\item{design}{List containing design matrix and several additional information like, 
e.g., number and names of covariates.} 
\item{Y}{Response object.} 
\item{penalty}{List containing all penalty matrices and some further information on penalties.} 
\item{response}{Vector containing 0-1 coded
response.} 
\item{X}{X matrix containing subject-specific covariates.} 
\item{Z1}{Z1 matrix containing subject-object-specific covariates.} 
\item{Z2}{Z2 matrix containing (subject)-object-specific covariates.} 
\item{lambda}{Vector of tuning parameters.} 
\item{control}{Control argument, specified by \code{\link{ctrl.BTLLasso}}.}
}
\description{
Performs BTLLasso, a method to model heterogeneity in paired comparison
data. Different types of covariates are allowd to have an influence on the
attractivity/strength of the objects. Covariates can be subject-specific, 
object-specific or subject-object-specific. L1 penalties are used to reduce the 
complexiy of the model by enforcing clusters of equal effects or by elimination of irrelevant
covariates.
}
\examples{

\dontrun{
##### Example with simulated data set containing X, Z1 and Z2
data(SimData)

## Specify tuning parameters
lambda <- exp(seq(log(151),log(1.05),length=30))-1

## Specify control argument, allow for object-specific order effects and penalize intercepts
ctrl <- ctrl.BTLLasso(penalize.intercepts = TRUE, object.order.effect = TRUE,
                      penalize.order.effect.diffs = TRUE)

## Simple BTLLasso model for tuning parameters lambda
m.sim <- BTLLasso(Y = SimData$Y, X = SimData$X, Z1 = SimData$Z1, 
                  Z2 = SimData$Z2, lambda = lambda, control = ctrl)

singlepaths(m.sim, x.axis = "loglambda")

## Cross-validate BTLLasso model for tuning parameters lambda
set.seed(5)
m.sim.cv <- cv.BTLLasso(Y = SimData$Y, X = SimData$X, Z1 = SimData$Z1, 
                        Z2 = SimData$Z2, lambda = lambda, control = ctrl)


singlepaths(m.sim.cv, x.axis = "loglambda", plot.order.effect = FALSE, plot.intercepts = FALSE, 
            plot.Z2 = FALSE)
paths(m.sim.cv, y.axis="L2")

## Example for bootstrap confidence intervals for illustration only
## Don't calculate bootstrap confidence intervals with B = 10
set.seed(5)
m.sim.boot <- boot.BTLLasso(m.sim.cv, B = 10, cores = 10)
ci.BTLLasso(m.sim.boot)

##### Example with small version from GLES data set
data(GLESsmall)

## vector of subtitles, containing the coding of the X covariates
subs.X <- c("","female (1); male (0)")

## vector of tuning parameters
lambda <- exp(seq(log(61),log(1),length=30))-1


## compute BTLLasso model 
m.gles <- BTLLasso(Y = GLESsmall$Y, X = GLESsmall$X, Z1 = GLESsmall$Z1, lambda = lambda)

singlepaths(m.gles, x.axis = "loglambda", subs.X = subs.X)
paths(m.gles, y.axis="L2")

## Cross-validate BTLLasso model 
m.gles.cv <- cv.BTLLasso(Y = GLESsmall$Y, X = GLESsmall$X, Z1 = GLESsmall$Z1, lambda = lambda)

singlepaths(m.gles.cv, x.axis = "loglambda", subs.X = subs.X)
}

}
\author{
Gunther Schauberger\cr \email{gunther@stat.uni-muenchen.de}\cr
\url{http://www.statistik.lmu.de/~schauberger/}
}
\references{
Schauberger, Gunther and Tutz, Gerhard (2015): Modelling
Heterogeneity in Paired Comparison Data - an L1 Penalty Approach with an
Application to Party Preference Data, \emph{Department of Statistics, LMU
Munich}, Technical Report 183
}
\seealso{
\code{\link{cv.BTLLasso}}, \code{\link{boot.BTLLasso}}, \code{\link{ctrl.BTLLasso}},
\code{\link{singlepaths}}, \code{\link{paths}}
}
\keyword{BTLLasso}

