% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/plotting_functions.R
\name{plot_params}
\alias{plot_params}
\title{Draw image plot of posterior model probability surface.}
\usage{
plot_params(
  x = NULL,
  y = NULL,
  z = NULL,
  t_levels = NULL,
  Pmc = 0.5,
  event = 1,
  k = 100,
  dlim = c(1e-04, 5),
  glim = c(1e-04, 5),
  zlim = c(0, 1),
  return_z = FALSE,
  epsilon = .Machine$double.eps,
  contours_only = FALSE,
  main = "Posterior Model Probability of Calibration",
  xlab = "delta",
  ylab = "gamma",
  optim_options = NULL,
  imgplt_options = list(legend.lab = ""),
  contour_options = list(drawlabels = TRUE, labcex = 0.6, lwd = 1, col =
    ifelse(contours_only, "black", "white"))
)
}
\arguments{
\item{x}{a numeric vector of predicted probabilities of an event. Must only
contain values in [0,1].}

\item{y}{a vector of outcomes corresponding to probabilities in \code{x}. Must
only contain two unique values (one for "events" and one for "non-events").
By default, this function expects a vector of 0s (non-events) and 1s
(events).}

\item{z}{Matrix returned by previous call to \code{plot_params()} containing
posterior model probabilities across k\eqn{\times}k grid of \eqn{\delta}
and \eqn{\gamma}. Assumes \code{z} was constructed using the same \code{k}, \code{dlim},
and \code{glim} as the current function call.}

\item{t_levels}{Vector of desired level(s) of calibration at which to plot
contours.}

\item{Pmc}{The prior model probability for the calibrated model \eqn{M_c}.}

\item{event}{Value in \code{y} that represents an "event".  Default value is 1.}

\item{k}{The number of uniformly spaced \eqn{\delta} and \eqn{\gamma} values
used to construct the k\eqn{\times}k grid.}

\item{dlim}{Vector with bounds for \eqn{\delta}, must be finite.}

\item{glim}{Vector with bounds for \eqn{\gamma}, must be finite.}

\item{zlim}{Vector with bounds for posterior probability of calibration, must
be in [0,1].}

\item{return_z}{Logical.  If \code{TRUE}, the matrix of posterior model
probabilities across the specified k\eqn{\times}k grid of \eqn{\delta} and
\eqn{\gamma} will be returned.}

\item{epsilon}{Amount by which probabilities are pushed away from 0 or 1
boundary for numerical stability. If a value in \code{x} < \code{epsilon}, it will be
replaced with \code{epsilon}.  If a value in \code{x} > \code{1-epsilon}, that value will
be replaced with \code{1-epsilon}.}

\item{contours_only}{Logical.  If \code{TRUE}, only the contours at the specified
\code{t_levels} will be plotted with no color for the posterior model
probability across the k\eqn{\times}k grid of \eqn{\delta} and
\eqn{\gamma}.}

\item{main}{Plot title.}

\item{xlab}{Label for x-axis.}

\item{ylab}{Label for x-axis.}

\item{optim_options}{List of additional arguments to be passed to \link[stats]{optim}().}

\item{imgplt_options}{List of additional arguments to be passed to \link[fields]{image.plot}().}

\item{contour_options}{List of additional arguments to be passed to \link[graphics]{contour}().}
}
\value{
If \code{return_z = TRUE}, a list with the following attributes is
returned: \item{\code{z}}{Matrix containing posterior model probabilities
across k\eqn{\times}k grid of uniformly spaced values of \eqn{\delta} and \eqn{\gamma}
in the specified ranges \code{dlim} and \code{glim}, respectively.}
\item{\code{dlim}}{Vector with bounds for
\eqn{\delta} used to construct z.}
\item{\code{glim}}{Vector with bounds for \eqn{\gamma} used to construct
z.}
\item{\code{k}}{The number of uniformly spaced \eqn{\delta} and \eqn{\gamma}
values used to construct z}
}
\description{
Function to visualize the posterior model probability of the given set of
probabilities, \code{x}, after LLO-adjustment via a grid of uniformly spaced set
of \eqn{\delta} and \eqn{\gamma} values with optional contours.
}
\details{
This function leverages the \link[fields]{image.plot} function from the
\link[fields]{fields} package and the \link[graphics]{contour} function from
the \link[graphics]{graphics} package.

The goal of this function is to visualize how the posterior model probability
changes under different recalibration parameters, as this is used in
boldness-recalibration.  To do so, a \code{k} by \code{k} grid of uniformly spaced
potential values for \eqn{\delta} and \eqn{\gamma} are constructed.  Then \code{x}
is LLO-adjusted under each pair of \eqn{\delta} and \eqn{\gamma} values. The
posterior model probability of each LLO-adjusted set is calculated and this
is the quantity we use to color each grid cell in the image plot to visualize
change in calibration.  See below for more details on setting the grid.

By default, only the posterior model probability surface is plotted. Argument
\code{t_levels} can be used to optionally add contours at specified levels of the
posterior model probability of calibration.   The goal of this is to help
visualize different values of \eqn{t} at which they may want to
boldness-recalibrate. To only draw the contours without the colored posterior
model probability surface, users can set \code{contours_only=TRUE}.
}
\section{Setting grid for \eqn{\delta} and \eqn{\gamma}}{


Arguments \code{dlim} and \code{glim} are used to set the bounds of the \eqn{\delta},
\eqn{\gamma} grid and the size is dictated by argument \code{k}. Some care is
required for the selection of these arguments. The goal is to determine
what range of \eqn{\delta} and \eqn{\gamma} encompasses the region of
non-zero posterior probabilities of calibration.  However, it is not
feasible to check the entire parameter space (as it is unbounded) and even
at smaller regions it can be difficult to detect the region in which
non-zero posterior probabilities are produced without as very dense grid
(large \code{k}), as the region is often quite small relative to the entire
parameter space. This is problematic, as computation time increases as \code{k}
grows.

We suggest the following scheme setting \code{k}, \code{dlim}, and \code{glim}. First, fix
\code{k} at some small number, less than 20 for sake of computation time. Then,
center a grid with small range around the MLEs for \eqn{\delta} and
\eqn{\gamma} for the given \code{x} and \code{y}. Increase the size of \code{k} until your
grid detects approximated the probability of calibration at the MLEs that
you expect. Then, expand your grid until it the region with high
probability of calibration is covered or contract your grid to "zoom in" on
the region. Then, increase \code{k} to create a fine grid of values.

Additionally, we caution users from including \eqn{\gamma = 0} in the grid.
This setting recalibrates all values in \code{x} to a single value which is not
desirable in practice.  Unless the single value is near the base rate, the
set will be poorly calibrated and minimally bold, which does not align with
the goal of boldness-recalibration.
}

\section{Reusing matrix \code{z} via \code{return_z}}{


The time bottleneck for this function occurs when calculating the posterior
model probabilities across the grid of parameter values. Thus it can be
useful to save the resulting matrix of values to be re-used to save time
when making minor cosmetic changes to your plot. If these adjustments do
not change the grid bounds or density, users can set \code{return_z=TRUE} to
return the underlying matrix of posterior mode probabilities for plotting.
Then, instead of specifying \code{x} and \code{y} users can just pass the returned
matrix as \code{z}.  Note this assumes you are NOT making any changes to \code{k},
\code{dlim}, or \code{glim}.  Also, it is not recommended that you construct your
own matrix to pass via \code{z} as this function relies on the structure as
returned by a previous call of \code{plot_params()}.
}

\section{Thinning}{


Another approach to speed up the calculations of this function is to thin
the data used. However, this is generally not recommended unless the sample
size is very large as the calculations of the posterior model probability
may change drastically under small sample sizes.  This can lead to
misleading results. Under large sample sizes where thinning is used, note
this is only an approximate visual of the posterior model probability.
}

\section{Grid cells that show up white / inaccuracies warning message}{


In some cases, grid cells in the plot may show up as white instead of one
of the colors from red to blue shown on the legend.  A white grid cell
indicates that there is no calculated posterior model probability at that
cell. There are two common reasons for this: (1) that grid cell location is
not covered by the \code{z} matrix used (i.e. you've adjusted the bounds without
recalculating z) or (2) the values of the parameters at these locations
cause the values in \code{x} to be LLO-adjusted such that they virtually equal 0
or 1.  This invokes the use of \code{epsilon} to push them away from these
boundaries for stability.   This typically happens when |gamma| is very large.
However, in these extreme cases this can cause inaccuracies in this plot.
For this reason, we either throw the warning message: "Probs too close to 0
or 1 under very large |gamma|" and allow the cell to be plotted as white
to notify the user and avoid plotting artifacts.

Additionally, when gamma is very close to 0, we cannot directly calculate
the MLEs for the grid shifted prediction and thus must use \code{optim()}
to approximate them.  In this case, we throw a \code{warning} to notify users there
may be inaccuracies.
}

\examples{

# Simulate 50 predicted probabilities
set.seed(49)
x <- runif(50)
# Simulated 50 binary event outcomes using x
y <- rbinom(50, 1, x)  # By construction, x is well calibrated.

#' # Set grid density k=20
plot_params(x, y, k=20)

# Adjust bounds on delta and gamma
plot_params(x, y, k=20, dlim=c(0.001, 3), glim=c(0.01,2))

# Increase grid density via k & save z matrix for faster plotting
zmat_list <- plot_params(x, y, k=100, dlim=c(0.001, 3), glim=c(0.01,2), return_z=TRUE)

# Reuse z matrix
plot_params(z=zmat_list$z, k=100, dlim=c(0.001, 3), glim=c(0.01,2))

# Add contours at t=0.95, 0.9, and 0.8
plot_params(z=zmat_list$z, k=100, dlim=c(0.001, 3), glim=c(0.01,2), t_levels=c(0.95, 0.9, 0.8))

# Add points for 95\% boldness-recalibration parameters
br95 <- brcal(x, y, t=0.95, print_level=0)
plot_params(z=zmat_list$z, k=100, dlim=c(0.001, 3), glim=c(0.01,2), t_levels=c(0.95, 0.9, 0.8))
points(br95$BR_params[1], br95$BR_params[2], pch=19, col="white")

# Change color and size of contours
plot_params(z=zmat_list$z, k=100, dlim=c(0.001, 3), glim=c(0.01,2), t_levels = c(0.99, 0.1), 
contour_options=list(col="orchid", lwd=2))

# Plot contours only
plot_params(z=zmat_list$z, k=100, dlim=c(0.001, 3), glim=c(0.01,2), t_levels=c(0.95, 0.9, 0.8),
contours_only=TRUE)

# Pass arguments to image.plot()
plot_params(z=zmat_list$z, k=100, dlim=c(0.001, 3), glim=c(0.01,2),
            imgplt_options=list(horizontal = TRUE, nlevel=10, 
            legend.lab="Posterior Model Prob"))

# See vignette for more examples

}
\references{
Guthrie, A. P., and Franck, C. T. (2024) Boldness-Recalibration
for Binary Event Predictions, \emph{The American Statistician} 1-17.

Nychka, D., Furrer, R., Paige, J., Sain, S. (2021). fields: Tools for
spatial data. R package version 15.2,
\url{https://github.com/dnychka/fieldsRPackage}.
}
