library(Seurat)
library(ggplot2)
library(ggforce)
library(dplyr)
library(data.table)
library(R6)
library(utils)

#' Visualize T cell clonal expansion with a ball-packing plot.
#'
#' Integrates a cell ranger T cell library into a Seurat object with a UMAP
#' reduction. Then gets sizes of unique clones and utilizes a circle-packing
#' algorithm to pack circles representing individual clones in approximately
#' the same UMAP coordinates and clusters into a ggplot object. 
#'
#' @param seurat_obj Seurat object with at least a UMAP reduction. Can either already have been integrated with a T cell library via `integrate_tcr(seurat_obj, tcr_df)`, in which case the subsequent `tcr_df` argument can be left empty. Else, the object must be a regular seurat object and a T cell library must be inputted in the following `tcr_df` argument
#' @param tcr_df If left empty, `seurat_obj` is assumed to be already integrated. Otherwise, should be a `data.frame` of the T cell library generated by 10X genomics' Cell Ranger. The dataframe has to at least have the `barcode` and `raw_clonotype_id` columns. 
#' @param clone_scale_factor numeric. Decides how much to scale each circle. Usually should be kept at around 0.01 to somewhat maintain UMAP structure for large datasets. However, if the plot that is displayed is ever blank, first try increasing this value. 
#' @param rad_scale_factor numeric. indicates how much the radii of the clones should decrease to add a slight gap between all of them. Defaults to 1 but 0.85-0.95 values are recommended. Both `rad_scale_factor` and `clone_scale_factor` may need to be repeatedly readjusted
#' @param res The number of points on the generated path per full circle. From plot viewers, if circles seem slightly too pixelated, it is highly recommended to first try to export the plot as an `.svg` before increasing `res`
#' @param ORDER logical. Decides if the largest clones should be at the cluster centroids
#' @param try_place If `TRUE`, always minimizes distance from a newly placed circle to the origin
#' @param verbose logical. Decides if visual cues print to the R console of the packing progress
#' @param repulse If `TRUE`, will attempt to push overlapping clusters away from each other.
#' @param repulsion_threshold numeric. The radius that cluster overlap is acceptable
#' @param repulsion_strength numeric. The smaller the value the less the clusters repulse each other
#' @param max_repulsion_iter numeric. The number of repulsion iterations, note that increasing this value may occasionally even lead to worse looking plots as clusters may repulse eachother too much
#' @param use_default_theme If `TRUE`, the resulting plot will have the same theme as the seurat UMAP. Else, the plot will simply have a blank background
#' @param show_origin logical. If `TRUE`, only the centers of each circle will be plotted
#' @param retain_axis_scales If `TRUE`, approximately maintains the axis scales of the original UMAP. However, it will only attempt to extend the axes and never shorten.
#' @param add_size_legend If `TRUE`, adds a legend to the plot titled `"Clone sizes"` indicating the relative sizes of clones. 
#' @param legend_sizes numeric vector. Indicates the circle sizes to be displayed on the legend and defaults to `c(1, 5, 10)`. 
#' @param legend_position character. Can be set to either `"top_left"`, `"top_right"`, `"bottom_left"`, `"bottom_right"` and places the legend roughly in the corresponding position
#' @param legend_buffer numeric. Indicates how much to "push" the legend towards the center of the plot from the selected corner. If negative, will push away
#' @param legend_color character. Indicates the hex color of the circles displayed on the legend. Defaults to the hex code for gray
#' @param legend_spacing numeric. Indicates the horizontal distance between each stacked circle on the size legend. Usually should be kept below 0.75 -ish depending on plot size.
#'
#' @return Returns a ggplot2 object of the ball packing plot. Can be operated on like normal ggplot objects
#'
#' @details Check out the web-only user vignette at
#' `https://qile0317.github.io/APackOfTheClones/articles/web_only/Clonal_expansion_plotting.html`
#' for a walkthrough on using this function, and additional details. 
#'
#' @seealso \code{\link{integrate_tcr}}
#'
#' @export
#'
#' @examples
#' library(Seurat)
#' library(APackOfTheClones)
#' data("mini_clonotype_data","mini_seurat_obj")
#'
#' # produce and show the ball-packing plot by integrating the data
#' ball_pack_plot <- clonal_expansion_plot(mini_seurat_obj, mini_clonotype_data)
#' ball_pack_plot
#' 
#' # it's also possible to input an integrated Seurat object
#' integrated_seurat_object <- integrate_tcr(mini_seurat_obj, mini_clonotype_data)
#' ball_pack_plot <- clonal_expansion_plot(integrated_seurat_object)
#' ball_pack_plot
#'
clonal_expansion_plot <- function(
  seurat_obj, tcr_df = "seurat_obj_already_integrated",
  res = 360,
  clone_scale_factor = 0.1, # do 0.5 for test ds - need to make an estimator based on testing
  rad_scale_factor = 0.95, 
  ORDER = TRUE,
  try_place = FALSE,
  verbose = TRUE,
  repulse = FALSE,
  repulsion_threshold = 1,
  repulsion_strength = 1,
  max_repulsion_iter = 10,
  use_default_theme = TRUE,
  show_origin = FALSE,
  retain_axis_scales = FALSE,
  add_size_legend = TRUE,
  legend_sizes = c(1, 5, 50),
  legend_position = "top_left",
  legend_buffer = 1.5,
  legend_color = "#808080",
  legend_spacing = 0.4) {

  # errors/warnings:
  if (is.null(seurat_obj@reductions[["umap"]])) {stop("No UMAP reduction found on the seurat object")}
  if ((!is.data.frame(tcr_df)) && is.null(seurat_obj@meta.data[["raw_clonotype_id"]])) {
    stop("Seurat object is missing the raw_clonotype_id data or isn't integrated with the TCR library. Consider integrating the T-cell library into the seurat object again.")
  }
  if (max_repulsion_iter > 1000) {
    warning("Repulsion iteration count is high, consider reducing max_repulsion_iter if runtime is too long")
  }

  # integrate TCR and count clonotypes
  if (is.data.frame(tcr_df)) {
    seurat_obj <- integrate_tcr(seurat_obj, tcr_df, verbose = verbose)
  }
  
  clone_size_list <- get_clone_sizes(seurat_obj, scale_factor = clone_scale_factor)
  centroid_list <- get_cluster_centroids(seurat_obj)

  # pack the plot
  result_plot <- plot_API(sizes = clone_size_list,
                          centroids = centroid_list,
                          num_clusters = count_umap_clusters(seurat_obj),
                          rad_decrease = rad_scale_factor,
                          ORDER = ORDER,
                          try_place = try_place,
                          progbar = verbose,
                          repulse = repulse,
                          thr = repulsion_threshold,
                          G = repulsion_strength,
                          max_repulsion_iter = max_repulsion_iter,
                          n = res,
                          origin = show_origin)
  
  #set theme
  if (use_default_theme) {
    result_plot <- result_plot +
      ggplot2::theme_classic() +
      ggplot2::xlab("UMAP_1") +
      ggplot2::ylab("UMAP_2") +
      ggplot2::ggtitle("Sizes of clones within each cluster")
  } else {
    result_plot <- result_plot + ggplot2::theme_void()
  }
  
  # retain axis scales on the resulting plot. The function sucks tho
  if (retain_axis_scales) {
    result_plot <- suppressMessages(invisible(retain_scale(seurat_obj, result_plot)))
  }
  
  if (verbose) {message("\nPlotting completed successfully")}
  if (add_size_legend) {
    return(insert_legend(
      plt = result_plot, circ_scale_factor = clone_scale_factor, sizes = legend_sizes,
      pos = legend_position, buffer = legend_buffer, color = legend_color, n = res,
      spacing = legend_spacing
      )
    )
  }
  result_plot
}

#' change the axis scales to fit the original plot approximately. Looks pretty bad atm.
#' A more advanced version could multiply axses by a small amount to retain ratios exactly
#' @importFrom ggplot2 coord_cartesian
#' @noRd
retain_scale <- function(seurat_obj, ball_pack_plt, buffer = 0) {
  
  test_umap_plt <- Seurat::DimPlot(object = seurat_obj, reduction = "umap")
  
  # get current ranges
  umap_xr <- ggplot2::ggplot_build(test_umap_plt)$layout$panel_scales_x[[1]]$range$range
  umap_yr <- ggplot2::ggplot_build(test_umap_plt)$layout$panel_scales_y[[1]]$range$range
  
  rm("test_umap_plt")
  
  ball_pack_xr <- ggplot2::ggplot_build(ball_pack_plt)$layout$panel_scales_x[[1]]$range$range
  ball_pack_yr <- ggplot2::ggplot_build(ball_pack_plt)$layout$panel_scales_y[[1]]$range$range
  
  # set new ranges
  min_xr <- min(ball_pack_xr[1], umap_xr[1]) - buffer
  max_xr <- max(ball_pack_xr[2], umap_xr[2]) + buffer
  
  min_yr <- min(ball_pack_yr[1], umap_yr[1]) - buffer
  max_yr <- max(ball_pack_yr[2], umap_yr[2]) + buffer
  
  return(ball_pack_plt + coord_cartesian(
    xlim = c(min_xr, max_xr),
    ylim = c(min_yr, max_yr)
    )
  )
}
