#' Gather Draft Picks
#'
#' Given a draft ID, gather draft picks (along with other metadata concerning the picks).
#'
#' @return Returns a data frame containing information about the draft picks.
#' @author Nick Bultman, \email{njbultman74@@gmail.com}, December 2021
#' @keywords draft picks
#' @importFrom httr GET content
#' @importFrom jsonlite fromJSON
#' @importFrom dplyr left_join
#' @export
#' @examples
#' \dontrun{get_draft_picks(688281872463106048)}
#'
#' @param draft_id Draft ID generated by Sleeper (numeric or character)
#'
get_draft_picks <- function(draft_id) {
  # Send request to API given draft ID specified
  x <- jsonlite::fromJSON(httr::content(httr::GET(paste0("https://api.sleeper.app/v1/draft/", draft_id, "/picks")), as = "text"))
  # Check class of data returned from request
  if(is.null(x)) {
    # If NULL is returned, no data was found - inform user and do not return anything
    message("No data found - was the draft ID entered correctly?")
  } else {
    # If NULL is not returned, strip out metadata nested data frame
    x_metadata <- x$metadata
    # Drop metadata nested data frame from main query
    x$metadata <- NULL
    # Join metadata data frame to main query by player ID
    x_fin <- dplyr::left_join(x, x_metadata, by = "player_id")
    # Return final data frame
    return(x_fin)
  }
}
