#' Plot method for `summary.seroincidence.by` objects
#'
#' @param object a `summary.seroincidence.by` object (generated by applying the `summary()` method to the output of [est.incidence.by()]).
#' @param xvar the name of a stratifying variable in `object`
#' @param alpha transparency for the points in the graph (1 = no transparency, 0 = fully transparent)
#' @param shape shape argument for `geom_point()`
#' @param width width for jitter
#' @param ... unused
#'
#' @return a [ggplot2::ggplot()] object
#' @export
#' @examples
#'
#' library(dplyr)
#' library(ggplot2)
#'
#' xs_data <-
#'   sees_pop_data_pk_100
#'
#' curve <-
#'   typhoid_curves_nostrat_100 %>%
#'   filter(antigen_iso %in% c("HlyE_IgA", "HlyE_IgG"))
#'
#' noise <-
#'   example_noise_params_pk
#'
#' est2 <- est.incidence.by(
#'   strata = c("catchment"),
#'   pop_data = xs_data,
#'   curve_params = curve,
#'   noise_params = noise,
#'   antigen_isos = c("HlyE_IgG", "HlyE_IgA"),
#'   #num_cores = 8 #Allow for parallel processing to decrease run time
#' )
#'
#' est2sum <- summary(est2)
#'
#' autoplot(est2sum, "catchment")
#'
autoplot.summary.seroincidence.by <- function(
    object,
    xvar,
    alpha = .7,
    shape = 1,
    width = 0.001,
    ...) {
  object %>%
    ggplot2::ggplot(
      ggplot2::aes(
        x = get(xvar),
        y = .data$incidence.rate
      )
    ) +
    ggplot2::geom_jitter(
      width = width,
      height = 0,
      aes(
        col = .data$nlm.convergence.code
      ),
      shape = shape,
      alpha = alpha
    ) +
    ggplot2::xlab(xvar) +
    ggplot2::ylab("Estimated incidence rate") +
    ggplot2::theme_linedraw() +
    ggplot2::expand_limits(x = 0, y = 0) +
    ggplot2::labs(col = "`nlm()` convergence code") +
    ggplot2::theme(legend.position = "bottom")
}
