% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/do_test.R
\name{do_test}
\alias{do_test}
\title{Do a Test on Each Replication}
\usage{
do_test(
  sim_all,
  test_fun,
  test_args = list(),
  map_names = c(fit = "fit"),
  results_fun = NULL,
  results_args = list(),
  parallel = FALSE,
  progress = FALSE,
  ncores = max(1, parallel::detectCores(logical = FALSE) - 1)
)
}
\arguments{
\item{sim_all}{The output of
\code{\link[=sim_out]{sim_out()}}.}

\item{test_fun}{A function to do the
test. See 'Details' for the requirement
of this function. There are some
built-in test functions in \code{power4mome},
described in 'Details'.}

\item{test_args}{A list of arguments
to be passed to the \code{test_fun}
function. Default is \code{list()}.}

\item{map_names}{A named character
vector specifying how the content of
the element \code{extra} in
each replication of \code{sim_all} map
to the argument of \code{test_fun}.
Default is \code{c(fit = "fit")},
indicating that the element \code{fit}
in the element \code{extra} is set to
the argument \code{fit} of \code{test_fun}.
That is, for the first replication,
\code{fit = sim_out[[1]]$extra$fit} when
calling \code{test_fun}.}

\item{results_fun}{The function to be
used to extract the test results.
See \code{Details} for the requirements
of this function. Default is \code{NULL},
assuming that the output of
\code{test_fun} can be used directly.}

\item{results_args}{A list of
arguments to be passed to the
\code{results_fun} function. Default is
\code{list()}.}

\item{parallel}{If \code{TRUE}, parallel
processing will be used to do the
tests. Default is \code{FALSE}.}

\item{progress}{If \code{TRUE}, the progress
of tests will be displayed.
Default is `FALSE.}

\item{ncores}{The number of CPU
cores to use if parallel processing
is used.}
}
\value{
An object of the class \code{test_out},
which is a list of length equal to
\code{sim_all}, one element for each
replication. Each element of the list
has two elements:
\itemize{
\item \code{test}: The output of the function
set to \code{test_fun}.
\item \code{test_results}: The output of
the function set to \code{results_fun}.
}
}
\description{
Do a test on each
replication in the output of
\code{\link[=sim_out]{sim_out()}}.
}
\details{
The function \code{\link[=do_test]{do_test()}} does
an arbitrary test in each
replication using the function set to
\code{test_fun}.
}
\section{The role of \code{do_test()}}{
The function \code{\link[=do_test]{do_test()}} is used by
the all-in-one function
\code{\link[=power4test]{power4test()}}. Users usually do not
call this function directly, though
developers can use this function to
develop other functions for power
analysis, or to build their own
workflows to do the power analysis.
}

\section{Major Test-Related Arguments}{
\subsection{The test function (test_fun)}{

The function set to \code{test_fun},
the \emph{test function}, usually
should work
on the output of \code{\link[lavaan:sem]{lavaan::sem()}},
\code{\link[lmhelprs:many_lm]{lmhelprs::many_lm()}}, or
\code{\link[stats:lm]{stats::lm()}}, but can also be a
function that works on the output
of the function set to \code{fit_function}
when calling \code{\link[=fit_model]{fit_model()}} or
\code{\link[=power4test]{power4test()}} (see \code{fit_model_args}).

The function has two default
requirements.

First, it has an argument \code{fit}, to
be set to the output of
\code{\link[lavaan:sem]{lavaan::sem()}} or another output
stored in the element \code{extra$fit} of
a replication in the \code{sim_all}
object. (This requirement can be
relaxed; see the section on \code{map_names}.)

That is, the function definition
should be of this from: \code{FUN(fit, ...)}. This is the form of all
\verb{test_*} functions in \code{power4mome}.

If other arguments are to be passed
to the test function, they can be set
to \code{test_args} as a named list.

Second, the test function must
returns an output that (a) can be
processed by the results function
(see below), or (b) is of
the required format for the output of
a results function (see the next
section). If it already returns
an output of the required format,
then there is no need to set the
results function.
}

\subsection{The results function (results_fun)}{

The test results will be extracted
from the output of \code{test_fun} by the
function set to \code{results_fun},
the \emph{results function}. If
the \code{test_fun} already returns an
output of the expected format
(see below), then set \code{results_fun}
to \code{NULL}, the default. The output
of \code{test_fun} will be used for
estimating power.

The function set to \code{results_fun}
must accept the output of \code{test_fun},
as the first argument, and return a
named list (which can be a data frame)
or a named vector with some
of the following
elements:
\itemize{
\item \code{est}: Optional. The estimate of a
parameter, if applicable.
\item \code{se}: Optional. The standard error
of the estimate, if applicable.
\item \code{cilo}: Optional. The lower limit of the
confidence interval, if applicable.
\item \code{cihi}: Optional. The upper limit of the
confidence interval, if applicable.
\item \code{sig}: Required. If \code{1}, the test is
significant. If \code{0}, the test is not
significant. If the test cannot be
done for any reason, it should be
\code{NA}.
}

The results can then be used to
estimate the power or Type I
error of the test.

For example, if
the null hypothesis is false, then
the proportion of significant, that
is, the mean of the values of \code{sig}
across replications, is the power.
}

\subsection{Built-in test functions}{

The package \code{power4mome} has some ready-to-use
test functions:
\itemize{
\item \code{\link[=test_indirect_effect]{test_indirect_effect()}}
\item \code{\link[=test_cond_indirect]{test_cond_indirect()}}
\item \code{\link[=test_cond_indirect_effects]{test_cond_indirect_effects()}}
\item \code{\link[=test_moderation]{test_moderation()}}
\item \code{\link[=test_index_of_mome]{test_index_of_mome()}}
\item \code{\link[=test_parameters]{test_parameters()}}
}

Please refer to their help pages
for examples.
}

\subsection{The argument map_names}{

This argument is for developers using
a test function that has a different
name for the argument of the fit
object (\code{"fit"}, the default).

If \code{test_fun} is set to a function
that works on an output of, say,
\code{\link[lavaan:sem]{lavaan::sem()}} but the argument name
for the output is not \code{fit}, the
mapping can be changed by
\code{map_names}.

For example,
\code{\link[lavaan:parameterEstimates]{lavaan::parameterEstimates()}}
receives an output of \code{\link[lavaan:sem]{lavaan::sem()}}
and reports the test results of model
parameters. However, the argument
name for the \code{lavaan} output is
\code{object.} To instruct \code{\link[=do_test]{do_test()}} to
do the test correctly when setting
\code{test_fun} to
\code{lavaan::parameterEstimates}, add
\code{map_names = c(object = "fit")}. The
element \code{fit} in a replication will
then set to the argument \code{object} of
\code{\link[lavaan:parameterEstimates]{lavaan::parameterEstimates()}}.
}
}

\section{The background for having the \code{results_fun} argument}{
In the early development of
\code{power4mome}, \code{test_fun} is designed
to accept existing functions from
other packages, such as
\code{\link[manymome:cond_indirect]{manymome::indirect_effect()}}. Their
outputs are not of the required
format for power analysis, and so
results functions are needed to
process their outputs. In the current
version of \code{power4mome}, some
ready-to-use test functions, usually
wrappers of these existing functions
from other packages, have been
developed, and they no longer need
results functions to process the
output. The argument \code{results_fun} is
kept for backward compatibility and
advanced users to use test functions
from other packages.
}

\examples{

# Specify the population model

mod <-
"
m ~ x
y ~ m + x
"

# Specify the effect sizes (population parameter values)

es <-
"
y ~ m: m
m ~ x: m
y ~ x: n
"

# Generate several simulated datasets

data_all <- sim_data(nrep = 5,
                     model = mod,
                     pop_es = es,
                     n = 100,
                     iseed = 1234)

# Fit the population model to each datasets

fit_all <- fit_model(data_all)

# Generate Monte Carlo estimates for forming Monte Carlo confidence intervals

mc_all <- gen_mc(fit_all,
                 R = 50,
                 iseed = 4567)

# Combine the results to a 'sim_all' object
sim_all <- sim_out(data_all = data_all,
                   fit = fit_all,
                   mc_out = mc_all)

# Test the indirect effect in each replication
# Set `parallel` to TRUE for faster, usually much faster, analysis
# Set `progress` to TRUE to display the progress of the analysis

test_all <- do_test(sim_all,
                    test_fun = test_indirect_effect,
                    test_args = list(x = "x",
                                     m = "m",
                                     y = "y",
                                     mc_ci = TRUE),
                    parallel = FALSE,
                    progress = FALSE)

# The result for each dataset
lapply(test_all, function(x) x$test_results)


}
\seealso{
See \code{\link[=power4test]{power4test()}} for
the all-in-one function that uses
this function. See \code{\link[=test_indirect_effect]{test_indirect_effect()}},
\code{\link[=test_cond_indirect]{test_cond_indirect()}},
\code{\link[=test_cond_indirect_effects]{test_cond_indirect_effects()}},
\code{\link[=test_moderation]{test_moderation()}},
\code{\link[=test_index_of_mome]{test_index_of_mome()}}, and
\code{\link[=test_parameters]{test_parameters()}} for examples of
test functions.
}
