% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/parttree.R
\name{parttree}
\alias{parttree}
\alias{parttree.rpart}
\alias{parttree._rpart}
\alias{parttree.workflow}
\alias{parttree.LearnerClassifRpart}
\alias{parttree.LearnerRegrRpart}
\alias{parttree.constparty}
\title{Convert a decision tree into a data frame of partition coordinates}
\usage{
parttree(tree, keep_as_dt = FALSE, flip = FALSE)
}
\arguments{
\item{tree}{An \code{\link[rpart]{rpart.object}} or alike. This includes
compatible classes from the \code{mlr3} and \code{tidymodels} frontends, or the
\code{constparty} class inheriting from \code{\link[partykit]{party}}.}

\item{keep_as_dt}{Logical. The function relies on \code{data.table} for internal
data manipulation. But it will coerce the final return object into a
regular data frame (default behavior) unless the user specifies \code{TRUE}.}

\item{flip}{Logical. Should we flip the "x" and "y" variables in the return
data frame? The default behaviour is for the first split variable in the
tree to take the "y" slot, and any second split variable to take the "x"
slot. Setting to \code{TRUE} switches these around.

\emph{Note:} This argument is primarily useful when it passed via
\link{geom_parttree} to ensure correct axes orientation as part of a \code{ggplot2}
visualization (see \link{geom_parttree} Examples). We do not expect users to
call \code{parttree(..., flip = TRUE)} directly. Similarly, to switch axes
orientation for the native (base graphics) \link{plot.parttree} method, we
recommend calling \code{plot(..., flip = TRUE)} rather than flipping the
underlying \code{parttree} object.}
}
\value{
A data frame comprising seven columns: the leaf node, its path, a
set of rectangle limits (i.e., xmin, xmax, ymin, ymax), and a final column
corresponding to the predicted value for that leaf.
}
\description{
Extracts the terminal leaf nodes of a decision tree that
contains no more that two numeric predictor variables. These leaf nodes are
then converted into a data frame, where each row represents a partition (or
leaf or terminal node) that can easily be plotted in 2-D coordinate space.
}
\examples{
library("parttree")
\dontshow{data.table::setDTthreads(2)}
#
## rpart trees

library("rpart")
rp = rpart(Kyphosis ~ Start + Age, data = kyphosis)

# A parttree object is just a data frame with additional attributes
(rp_pt = parttree(rp))
attr(rp_pt, "parttree")

# simple plot
plot(rp_pt)

# removing the (recursive) partition borders helps to emphasise overall fit
plot(rp_pt, border = NA)

# customize further by passing extra options to (tiny)plot
plot(
   rp_pt,
   border  = NA,                                     # no partition borders
   pch     = 16,                                     # filled points
   alpha   = 0.6,                                    # point transparency
   grid    = TRUE,                                   # background grid
   palette = "classic",                              # new colour palette
   xlab    = "Topmost vertebra operated on",         # custom x title
   ylab    = "Patient age (months)",                 # custom y title
   main    = "Tree predictions: Kyphosis recurrence" # custom title
)

#
## conditional inference trees from partyit

library("partykit")
ct = ctree(Species ~ Petal.Length + Petal.Width, data = iris)
ct_pt = parttree(ct)
plot(ct_pt, pch = 19, palette = "okabe", main = "ctree predictions: iris species")

## rpart via partykit
rp2 = as.party(rp)
parttree(rp2)

#
## various front-end frameworks are also supported, e.g.

# tidymodels

# install.packages("parsnip")
library(parsnip)

decision_tree() |>
  set_engine("rpart") |>
  set_mode("classification") |>
  fit(Species ~ Petal.Length + Petal.Width, data=iris) |>
  parttree() |>
  plot(main = "This time brought to you via parsnip...")

# mlr3 (NB: use `keep_model = TRUE` for mlr3 learners)

# install.packages("mlr3")
library(mlr3)

task_iris = TaskClassif$new("iris", iris, target = "Species")
task_iris$formula(rhs = "Petal.Length + Petal.Width")
fit_iris = lrn("classif.rpart", keep_model = TRUE) # NB!
fit_iris$train(task_iris)
plot(parttree(fit_iris), main = "... and now mlr3")

}
\seealso{
\link{plot.parttree}, \link{geom_parttree}, \code{\link[rpart]{rpart}},
\code{\link[partykit]{ctree}} \link[partykit:ctree]{partykit::ctree}.
}
