#' add_osm_objects
#'
#' Adds layers of spatial objects (polygons, lines, or points generated by
#' extract_osm_objects ()) to a graphics object initialised with
#' plot_osm_basemap().
#'
#' @param obj an sp SPDF or SLDF (list of polygons or lines) returned by
#' extract_osm_objects ()
#' @param col colour of polygons or lines (default = 'gray40')
#' @param border border colour of polygons
#' @param ... other parameters to be passed to polygons (such as border),
#' lines (such as lwd, lty), or points (such as pch, cex)
#' @return nothing (adds to graphics.device opened with plot_osm_basemap())
#' @export
#'
#' @examples
#' plot_osm_basemap (bbox=get_bbox (c (-0.15, 51.5, -0.1, 51.52)), col="gray20")
#' add_osm_objects (london$dat_BNR, col="gray40") # non-residential buildings

add_osm_objects <- function (obj=obj, col='gray40', border=NA, ...)
{
    if (is.null (dev.list ()))
        stop ('add_osm_objects can only be called after plot_osm_basemap')

    if (class (obj) == 'SpatialPolygonsDataFrame')
    {
        plotfunPts <- function (i, dx=dx, dy=dy, col=col, border=border, ...) 
        {
            xy <- slot (slot (i, 'Polygons') [[1]], 'coords')
            if (diff (range (xy [,1])) > dx | diff (range (xy [,2])) > dy)
                polypath (xy, col=col, border=border, ...)
        }
        # Find out which objects are < 1 pixel in size. NOTE this presumes
        # standard device resolution of 72dpi. 
        # TODO#1: Find out how to read dpi from open devices and modify
        din <- par ("din") * 72
        dx <- diff (par ("usr") [1:2]) / din [1]
        dy <- diff (par ("usr") [3:4]) / din [2]
        # NOTE dy=dx only if figures are sized automatically
        junk <- lapply (slot (obj, 'polygons'), function (i)
                    plotfunPts (i, dx=dx, dy=dy, col=col, border=border, ...))
    } else if (class (obj) == 'SpatialLinesDataFrame')
    {
        plotfunLines <- function (i, col=col, ...) 
        {
            xy <- slot (slot (i, 'Lines') [[1]], 'coords')
            lines (xy, col=col, ...)
        }
        junk <- lapply (slot (obj, 'lines'), function (i)
                        plotfunLines (i, col=col, ...))
    } else if (class (obj) == 'SpatialPointsDataFrame')
    {
        xy <- slot (obj, 'coords')
        points (xy[,1], xy[,2], col=col, ...)
    }
}

