% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cor.matrix.R
\name{cor.matrix}
\alias{cor.matrix}
\title{Correlation Matrix with Statistical Significance Testing}
\usage{
cor.matrix(x, method = c("pearson", "spearman", "kendall-b"),
           use = c("listwise", "pairwise"), group = NULL,
           print = c("all", "cor", "n", "p"), tri = c("both", "lower", "upper"),
           p.adj = c("none", "bonferroni", "holm", "hochberg",
                     "hommel", "BH", "BY", "fdr"),
           digits = 2, pval.digits = 3, as.na = NULL, check = TRUE,
           output = TRUE)
}
\arguments{
\item{x}{a matrix or data frame.}

\item{method}{a character vector indicating which correlation coefficient is to be computed, i.e.
\code{"pearson"} for Pearson product-moment correlation coefficient (default),
\code{"spearman"} for Spearman's rank-order correlation coefficient, or \code{kendall-b} for
Kendall's Tau-b correlation coefficient.}

\item{use}{a character vector giving a method for computing a correlation matrix in the presence
of missing values, i.e., \code{"listwise"} for listwise deletion and \code{"pairwise"} for
pairwise deletion}

\item{group}{a numeric vector, factor or character vector as grouping variable to show results for
each group separately, i.e., upper triangular for one group and lower triangular for
another group. Note that the grouping variable is limited to two groups.}

\item{print}{a character string or character vector indicating which additional results to show,
i.e. \code{"all"}, for all additional results: \code{"n"} for the sample sizes, and \code{"p"}
for \emph{p}-values.}

\item{tri}{a character string indicating which triangular of the matrix to show on the console,
i.e., \code{both} for upper and lower triangular, \code{lower} (default) for the
lower triangular, and \code{upper} for the upper triangular.}

\item{p.adj}{a character string indicating an adjustment method for multiple testing based on \code{\link{p.adjust}},
i.e.,  \code{none} (default), \code{bonferroni}, \code{holm}, \code{hochberg}, \code{hommel},
\code{BH}, \code{BY}, or \code{fdr}.}

\item{digits}{an integer value indicating the number of decimal places to be used for displaying
correlation coefficients.}

\item{pval.digits}{an integer indicating the number of decimal places to be used for displaying \emph{p}-values.}

\item{as.na}{a numeric vector indicating user-defined missing values,
i.e. these values are converted to \code{NA} before conducting the analysis.}

\item{check}{logical: if \code{TRUE}, argument specification is checked.}

\item{output}{logical: if \code{TRUE}, output is shown on the console.}
}
\value{
Returns an object of class \code{cor.matrix}, which is a list with following entries: function call (\code{call}),
matrix or data frame specified in \code{x} (\code{data}), specification of function arguments (\code{args}), and
list with results (\code{result}).
}
\description{
This function computes a correlation matrix and computes significance values (\emph{p}-values) for testing the hypothesis
H0: \eqn{\rho} = 0 for all possible pairs of variables.
}
\examples{
dat <- data.frame(group = c("a", "a", "a", "a", "a",
                            "b", "b", "b", "b", "b"),
                  x = c(5, NA, 6, 4, 6, 7, 9, 5, 8, 7),
                  y = c(3, 3, 5, 6, 7, 4, 7, NA, NA, 8),
                  z = c(1, 3, 1, NA, 2, 4, 6, 5, 9, 6))

# Pearson product-moment correlation coefficient matrix using pairwise deletion
cor.matrix(dat[, c("x", "y", "z")])

# Spearman's rank-order correlation matrix using pairwise deletion
cor.matrix(dat[, c("x", "y", "z")], method = "spearman")

# Kendall's Tau-b correlation matrix using pairwise deletion
cor.matrix(dat[, c("x", "y", "z")], method = "kendall-b")

# Pearson product-moment correlation coefficient matrix using pairwise deletion,
# print sample size and significance values
cor.matrix(dat[, c("x", "y", "z")], print = "all")

# Pearson product-moment correlation coefficient matrix using listwise deletion,
# print sample size and significance values
cor.matrix(dat[, c("x", "y", "z")], use = "listwise", print = "all")

# Pearson product-moment correlation coefficient matrix using listwise deletion,
# print sample size and significance values with Bonferroni correction
cor.matrix(dat[, c("x", "y", "z")], use = "listwise", print = "all", p.adj = "bonferroni")

# Pearson product-moment correlation coefficient matrix using pairwise deletion,
# results for group "a" and "b" separately
cor.matrix(dat[, c("x", "y", "z")], group = dat$group, print = "all")
}
\references{
Rasch, D., Kubinger, K. D., & Yanagida, T. (2011). \emph{Statistics in psychology - Using R and SPSS}.
John Wiley & Sons.
}
\seealso{
\code{\link{alpha.coef}}, \code{\link{cohens.d}}, \code{\link{cont.coef}}, \code{\link{cramers.v}},
\code{\link{multilevel.icc}}, \code{\link{phi.coef}}, \code{\link{na.auxiliary}}, \code{\link{size.cor}}.
}
\author{
Takuya Yanagida \email{takuya.yanagida@univie.ac.at}
}
