% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/quantifier.R
\name{quantifier}
\alias{quantifier}
\title{A quantifier is a function that computes a fuzzy truth value of a claim about
the quantity. This function creates the <1>-type quantifier. (See the examples
below on how to use it as a quantifier of the <1,1> type.)}
\usage{
quantifier(
  quantity = c("all", "almost.all", "most", "many", "some", "at.least"),
  n = NULL,
  alg = c("lukasiewicz", "goedel", "goguen")
)
}
\arguments{
\item{quantity}{the quantity to be evaluated. 'all' computes the degree of
truth to which all elements of the universe have the given property,
'almost.all', #'   'most', and 'many' evaluate whether the property is
present in extremely big, very big, or not small number of elements from
the universe, where these linguistic expressions are internally modeled
using the \code{\link[=lingexpr]{lingexpr()}} function. 'at.least' quantity requires the 'n'
argument to be specified, as it computes the truth value that at least
\eqn{n} elements from the universe have the given property.}

\item{n}{the number of elements in the 'at.least n' quantifier}

\item{alg}{the underlying algebra in which to compute the quantifier.
Note that the algebra must have properly defined the \code{order} function,
as in the case of 'goedel', 'goguen', or 'lukasiewicz' algebra, (see the
\code{\link[=algebra]{algebra()}} function) or as in the \code{\link[=dragonfly]{dragonfly()}} or \code{\link[=lowerEst]{lowerEst()}} algebra.}
}
\value{
A two-argument function, which expects two numeric vectors of equal length
(the vector elements are recycled to ensure equal lengths). The first argument, \code{x},
is a vector of membership degrees to be measured, the second argument, \code{w}, is
the vector of weights to which the element belongs to the universe.

Let \eqn{U} be the set of input vector indices (1 to \code{length(x)}). Then the quantifier
computes the truth values accordingly to the following formula:
\eqn{\vee_{z \subseteq U} \wedge_{u \in z} (x[u]) \wedge measure(m_z)},
where
\eqn{m_z = sum(w)} for \code{"some"} and \verb{"at.least} and \eqn{m_z = sum(w[z]) / sum(w)} otherwise.
See \code{\link[=sugeno]{sugeno()}} for more details on how the quantifier is evaluated.

Setting \code{w} to 1 yields to operation of the <1> quantifier as developed by Dvořák et al.
To compute the <1,1> quantifier as developed by Dvořák et al., e.g. "almost all A are B", \code{w} must
be set again to 1 and \code{x} to the result of the implication \eqn{A \Rightarrow B}.
To compute the <1,1> quantifier as proposed by Murinová et al., e.g. "almost all A are B",
\code{x} must be set to the result of the implication \eqn{A \Rightarrow B} and \code{w} to the membership
degrees of \eqn{A}. See the examples below.
}
\description{
A quantifier is a function that computes a fuzzy truth value of a claim about
the quantity. This function creates the <1>-type quantifier. (See the examples
below on how to use it as a quantifier of the <1,1> type.)
}
\examples{
  # Dvorak <1> "almost all" quantifier
  q <- quantifier('almost.all')
  a <- c(0.9, 1, 1, 0.2, 1)
  q(x=a, w=1)

  # Dvorak <1,1> "almost all" quantifier (w set to 1)
  a <- c(0.9, 1, 1, 0.2, 1)
  b <- c(0.2, 1, 0, 0.5, 0.8)
  q <- quantifier('almost.all')
  q(x=lukas.residuum(a, b), w=1)

  # Murinová <1,1> "almost all" quantifier (note w set to a)
  a <- c(0.9, 1, 1, 0.2, 1)
  b <- c(0.2, 1, 0, 0.5, 0.8)
  q <- quantifier('almost.all')
  q(x=lukas.residuum(a, b), w=a)

  # Murinová <1,1> "some" quantifier
  a <- c(0.9, 1, 1, 0.2, 1)
  b <- c(0.2, 1, 0, 0.5, 0.8)
  q <- quantifier('some')
  q(x=plukas.tnorm(a, b), w=a)
}
\references{
Dvořák, A., Holčapek, M. L-fuzzy quantifiers of type <1> determined by fuzzy measures.
Fuzzy Sets and Systems vol.160, issue 23, 3425-3452, 2009.

Dvořák, A., Holčapek, M. Type <1,1> fuzzy quantifiers determined by fuzzy measures.
IEEE International Conference on Fuzzy Systems (FuzzIEEE), 2010.

Murinová, P., Novák, V. The theory of intermediate quantifiers in fuzzy natural logic
revisited and the model of "Many". Fuzzy Sets and Systems, vol 388, 2020.
}
\seealso{
\code{\link[=sugeno]{sugeno()}}, \code{\link[=lingexpr]{lingexpr()}}
}
\author{
Michal Burda
}
\keyword{models}
\keyword{robust}
