#' Create HTML Report from irace data
#' 
#' This function creates an HTML report of the most relevant irace data.  This
#' report provides general statistics and plots that show the best
#' configurations and their performance. Example: <https://auto-optimization.github.io/iraceplot/articles/example/report_example.html>
#'
#' @template arg_irace_results
#'
#' @param filename (`character(1)`)
#' Filename indicating where to save the report (example: `"~/path-to/filename"`).
#' @param sections (`list()`) List of sections to enable/disable. This is useful for disabling sections that may cause problems, such as out-of-memory errors. `NA` means automatically enable/disable a section depending on the memory required.
#'
#' @template arg_interactive
#'
#  Useless imports to suppress a NOTE generated by "R CMD check" 
#' @importFrom DT renderDataTable
#' @importFrom knitr knit
#' 
#' @return filename where the report was created or it opens the report in the default browser (interactive).
#'
#' @examples
#' \donttest{ 
#'  withr::with_tempdir({
#'    iraceResults <- read_logfile(system.file(package="irace", "exdata",
#'                                             "irace-acotsp.Rdata", mustWork = TRUE))
#'    report(iraceResults, filename = file.path(getwd(), "report"))
#'  }, clean = !base::interactive())
#' }
#' @export
report <- function(irace_results, filename = "report",
                   sections = list(experiments_matrix = NULL, convergence = FALSE),
                   interactive = base::interactive())
{
  if (missing(irace_results)) stop("irace_results is required")
  # render() already checks this but the error is not clear enough.
  if (! rmarkdown::pandoc_available("1.12.3", error = FALSE))
    stop("pandoc version 1.12.3 or higher is required and was not found. ",
         "You can install the RStudio IDE, which has bundled a version of Pandoc. ",
         "Otherwise, follow the instructions at https://pandoc.org/installing.html .")
  
  irace_results <- irace::read_logfile(irace_results)

  # Large experiments matrix crashes pandoc.
  if (is.null(sections$experiments_matrix) || is.na(sections$experiments_matrix)) {
    sections$experiments_matrix <- (prod(dim(irace_results$experiments)) < 128L*1024L)
    if (!sections$experiments_matrix)
      iraceplot_warn("Race overview disable because the experiments matrix",
                     " is very large (set {.code sections$experiments_matrix=TRUE} to enable).")
  }
  
  filename <- irace::path_rel2abs(maybe_add_file_extension(filename, "html"))
  cli_alert_info("Creating file '{.file {filename}}'.\n")
  rmarkdown::render(input=system.file("template", "report_html.Rmd", package = "iraceplot"),
                    output_file=filename, clean = FALSE)
  if (interactive) utils::browseURL(filename)
  filename
}
