% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/micro_read_chunked.R
\name{read_ipums_micro_chunked}
\alias{read_ipums_micro_chunked}
\alias{read_ipums_micro_list_chunked}
\title{Read data from an IPUMS microdata extract by chunk}
\usage{
read_ipums_micro_chunked(
  ddi,
  callback,
  chunk_size = 10000,
  vars = NULL,
  data_file = NULL,
  verbose = TRUE,
  var_attrs = c("val_labels", "var_label", "var_desc"),
  lower_vars = FALSE
)

read_ipums_micro_list_chunked(
  ddi,
  callback,
  chunk_size = 10000,
  vars = NULL,
  data_file = NULL,
  verbose = TRUE,
  var_attrs = c("val_labels", "var_label", "var_desc"),
  lower_vars = FALSE
)
}
\arguments{
\item{ddi}{Either a path to a DDI .xml file downloaded from
\href{https://www.ipums.org/}{IPUMS}, or an
\link{ipums_ddi} object parsed by \code{\link[=read_ipums_ddi]{read_ipums_ddi()}}. See
\emph{Downloading IPUMS files} below.}

\item{callback}{An \link{ipums_callback} object, or a function
that will be converted to an \code{IpumsSideEffectCallback} object. Callback
functions should include both data (\code{x}) and position (\code{pos}) arguments.
See examples.}

\item{chunk_size}{Integer number of observations to
read per chunk. Higher values use more RAM, but
typically result in faster processing. Defaults to 10,000.}

\item{vars}{Names of variables to include in the output. Accepts a
vector of names or a \link[=selection_language]{tidyselect selection}.
If \code{NULL}, includes all variables in the file.

For hierarchical data, the \code{RECTYPE} variable is always included even if
unspecified.}

\item{data_file}{Path to the data (.gz) file associated with
the provided \code{ddi} file. By default, looks for the data file in the same
directory as the DDI file. If the data file has been moved, specify
its location here.}

\item{verbose}{Logical indicating whether to display IPUMS conditions and
progress information.}

\item{var_attrs}{Variable attributes from the DDI to add to the columns of
the output data. Defaults to all available attributes.
See \code{\link[=set_ipums_var_attributes]{set_ipums_var_attributes()}} for more details.}

\item{lower_vars}{If reading a DDI from a file,
a logical indicating whether to convert variable names to lowercase.
Defaults to \code{FALSE} for consistency with IPUMS conventions.

This argument will be ignored if argument \code{ddi} is
an \link{ipums_ddi} object. Use \code{\link[=read_ipums_ddi]{read_ipums_ddi()}} to convert variable
names to lowercase when reading a DDI file.

Note that if reading in chunks from a .csv or
.csv.gz file, the callback function will be called \emph{before} variable names
are converted to lowercase, and thus should reference uppercase variable
names.}
}
\value{
Depends on the provided callback object. See \link{ipums_callback}.
}
\description{
Read a microdata dataset downloaded from the IPUMS extract system in chunks.

Use these functions to read a file that is too large to store in memory
at a single time. The file is processed in chunks of a given size, with a
provided callback function applied to each chunk.

Two files are required to load IPUMS microdata extracts:
\itemize{
\item A \href{https://ddialliance.org/introduction-to-ddi}{DDI codebook} file
(.xml) used to parse the extract's data file
\item A data file (either .dat.gz or .csv.gz)
}

See \emph{Downloading IPUMS files} below for more information about downloading
these files.

\code{read_ipums_micro_chunked()} and \code{read_ipums_micro_list_chunked()} differ
in their handling of extracts that contain multiple record types.
See \emph{Data structures} below.

Note that Stata, SAS, and SPSS file formats are not supported by
ipumsr readers. Convert your extract to fixed-width or CSV format, or see
\href{https://haven.tidyverse.org/index.html}{haven} for help
loading these files.
}
\section{Data structures}{

Files from IPUMS projects that contain data for multiple types of records
(e.g. household records and person records) may be either rectangular
or hierarchical.

Rectangular data are transformed such that each row of data
represents only one type of record. For instance, each row will represent
a person record, and all household-level information for that person will
be included in the same row.

Hierarchical data have records of
different types interspersed in a single file. For instance, a household
record will be included in its own row followed by the person records
associated with that household.

Hierarchical data can be read in two different formats:
\itemize{
\item \code{read_ipums_micro_chunked()} reads each chunk of data into a
\code{\link[tibble:tbl_df-class]{tibble}} where each row represents a single record,
regardless of record type. Variables that do not apply to a particular
record type will be filled with \code{NA} in rows of that record type. For
instance, a person-specific variable will be missing in all rows
associated with household records. The provided \code{callback} function should
therefore operate on a \code{tibble} object.
\item \code{read_ipums_micro_list_chunked()} reads each chunk of data into a list of
\code{tibble} objects, where each list element contains
only one record type. Each list element is named with its corresponding
record type. The provided \code{callback} function should therefore operate
on a list object. In this case, the chunk size references the total
number of rows \emph{across} record types, rather than in each
record type.
}
}

\section{Downloading IPUMS files}{


You must download both the DDI codebook and the data file from the IPUMS
extract system to load the data into R. \verb{read_ipums_micro_*()} functions
assume that the data file and codebook share a common base file name and
are present in the same directory. If this is not the case, provide a
separate path to the data file with the \code{data_file} argument.

If using the IPUMS extract interface:
\itemize{
\item Download the data file by clicking \strong{Download .dat} under
\strong{Download Data}.
\item Download the DDI codebook by right clicking on the \strong{DDI} link in the
\strong{Codebook} column of the extract interface and selecting \strong{Save as...}
(on Safari, you may have to select \strong{Download Linked File as...}).
Be sure that the codebook is downloaded in .xml format.
}

If using the IPUMS API:
\itemize{
\item For supported collections, use \code{\link[=download_extract]{download_extract()}} to download a completed
extract via the IPUMS API. This automatically downloads both the DDI
codebook and the data file from the extract and
returns the path to the codebook file.
}
}

\examples{
suppressMessages(library(dplyr))

# Example codebook file
cps_rect_ddi_file <- ipums_example("cps_00157.xml")

# Function to extract Minnesota cases from CPS example
# (This can also be accomplished by including case selections
# in an extract definition)
#
# Function must take `x` and `pos` to refer to data and row position,
# respectively.
filter_mn <- function(x, pos) {
  x[x$STATEFIP == 27, ]
}

# Initialize callback
filter_mn_callback <- IpumsDataFrameCallback$new(filter_mn)

# Process data in chunks, filtering to MN cases in each chunk
read_ipums_micro_chunked(
  cps_rect_ddi_file,
  callback = filter_mn_callback,
  chunk_size = 1000,
  verbose = FALSE
)

# Tabulate INCTOT average by state without storing full dataset in memory
read_ipums_micro_chunked(
  cps_rect_ddi_file,
  callback = IpumsDataFrameCallback$new(
    function(x, pos) {
      x \%>\%
        mutate(
          INCTOT = lbl_na_if(
            INCTOT,
            ~ grepl("Missing|N.I.U.", .lbl)
          )
        ) \%>\%
        filter(!is.na(INCTOT)) \%>\%
        group_by(STATEFIP = as_factor(STATEFIP)) \%>\%
        summarize(INCTOT_SUM = sum(INCTOT), n = n(), .groups = "drop")
    }
  ),
  chunk_size = 1000,
  verbose = FALSE
) \%>\%
  group_by(STATEFIP) \%>\%
  summarize(avg_inc = sum(INCTOT_SUM) / sum(n))

# `x` will be a list when using `read_ipums_micro_list_chunked()`
read_ipums_micro_list_chunked(
  ipums_example("cps_00159.xml"),
  callback = IpumsSideEffectCallback$new(function(x, pos) {
    print(
      paste0(
        nrow(x$PERSON), " persons and ",
        nrow(x$HOUSEHOLD), " households in this chunk."
      )
    )
  }),
  chunk_size = 1000,
  verbose = FALSE
)

# Using the biglm package, you can even run a regression without storing
# the full dataset in memory
if (requireNamespace("biglm")) {
  lm_results <- read_ipums_micro_chunked(
    ipums_example("cps_00160.xml"),
    IpumsBiglmCallback$new(
      INCTOT ~ AGE + HEALTH, # Model formula
      function(x, pos) {
        x \%>\%
          mutate(
            INCTOT = lbl_na_if(
              INCTOT,
              ~ grepl("Missing|N.I.U.", .lbl)
            ),
            HEALTH = as_factor(HEALTH)
          )
      }
    ),
    chunk_size = 1000,
    verbose = FALSE
  )

  summary(lm_results)
}
}
\seealso{
\code{\link[=read_ipums_micro_yield]{read_ipums_micro_yield()}} for more flexible handling of large
IPUMS microdata files.

\code{\link[=read_ipums_micro]{read_ipums_micro()}} to read data from an IPUMS microdata extract.

\code{\link[=read_ipums_ddi]{read_ipums_ddi()}} to read metadata associated with an IPUMS microdata
extract.

\code{\link[=read_ipums_sf]{read_ipums_sf()}} to read spatial data from an IPUMS extract.

\code{\link[=ipums_list_files]{ipums_list_files()}} to list files in an IPUMS extract.
}
