% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/raxml.R
\name{raxml}
\alias{raxml}
\title{Maximum Likelihood Tree Estimation with RAxML}
\usage{
raxml(
  DNAbin,
  m = "GTRCAT",
  f,
  N,
  p,
  b,
  x,
  k,
  weights,
  partitions,
  outgroup,
  backbone = NULL,
  file = paste0("fromR_", Sys.Date()),
  exec,
  threads
)
}
\arguments{
\item{DNAbin}{A matrix of DNA sequences of class \code{\link{DNAbin}}.}

\item{m}{A vector of mode \code{"character"} defining a model of molecular
evolution; currently only GTR model available.}

\item{f}{A vector of mode \code{"character"} selecting an RAxML algorithm
analogous to the \code{-f} flag (see Detail section and RAxML manual).}

\item{N}{Either of mode \code{"integer"} or \code{"character"}. Integers give
the number of independent searches on different starting tree or replicates
in bootstrapping. Alternatively, one of four bootstopping criteria can be
chosen: \code{"autoFC"}, \code{"autoMR"}, \code{"autoMRE"}, or
\code{"autoMRE_IGN"}.}

\item{p}{Integer, setting a random seed for the parsimony starting trees.}

\item{b}{Integer, setting a random seed for bootstrapping.}

\item{x}{Integer, setting a random seed for rapid bootstrapping.}

\item{k}{Logical, if \code{TRUE}, the branch lengths of bootstrapped trees are
recorded.}

\item{weights}{A vector of mode \code{"numeric"} giving integers to assign
individual weights to each column of the alignment. (-a)}

\item{partitions}{A data frame giving the partitions of the alignment.}

\item{outgroup}{A vector of mode \code{"character"} containing the names of
the outgroup taxa.}

\item{backbone}{A \code{\link{phylo}} object representing a backbone tree.}

\item{file}{A vector of mode \code{"character"} giving a name for the output
files.}

\item{exec}{A vector of mode \code{"character"} giving the path to the
directory containing the RAxML executable. The default value will work on
Mac OS X if the folder containing the executable is renamed to
\code{"RAxML-8.0.3"}.}

\item{threads}{Integer, giving the number of parallel threads to use (PTHREADS
only).}
}
\value{
A list with a variable number of elements, depending on the analysis
 chosen: \tabular{ll}{ \code{"info"} \tab RAxML log file as character
 string\cr \code{"bestTree"} \tab MLE of tree\cr \code{"bipartitions"} \tab
 MLE of tree annotated with bootstrap proportions\cr \code{"bootstrap"} \tab
 bootstrapped trees\cr }
}
\description{
Provides an interface to the C program \bold{RAxML} (see
 Reference section) for maximum likelihood estimation of tree topology and/or
 branch lengths, rapid and conventional non-parametric bootstrapping, mapping
 splits onto individual topologies, and a lot more. See the RAxML manual for
 details, especially if you are a new user of RAxML.
}
\details{
There are some limitations of this wrapper compared to RAxML run
 directly from the command line. \enumerate{ \item Only DNA is allowed as
 data type. \item Option \code{f} can only take a limited number of values
 (\code{d}, \code{a}). } % close enumerate

 RAxML needs the specification of random seeds for parsimony estimation of
 starting trees and for bootstrap resampling. The corresponding argument
 names in \code{raxml} are identical to the flags used by RAxML (\code{-p},
 \code{-b}, and \code{-x}). If you choose not to give any values,
 \code{raxml} will generate a (different) value for each required random
 seed every time it is called. Be aware that \code{\link{set.seed}} will work
 only for \code{p}, but not for \code{b} or \code{x}.
}
\note{
RAxML is a C program and the source code is not contained in this
  package. This means that in order to run this function you will need to
  install RAxML yourself. See
  \url{https://cme.h-its.org/exelixis/web/software/raxml/} for the most recent
  documentation and source code of RAxML. Depending on where you chose to
  install RAxML, you need to adjust the \code{exec} argument.

 \code{raxml} was last tested and running fine on Mac OS X with RAxML 8.0.29.
 Please be aware that calling third-party software from within R is a
 platform-specific process and I cannot guarantee that \code{raxml} will
 behave properly on any system.
}
\examples{
## bark beetle sequences
data(ips.cox1)
data(ips.16S)
data(ips.28S)

ips <- cbind(ips.cox1, ips.16S, ips.28S,
            fill.with.gaps = TRUE)

exec <- "/Applications/RAxML-code/standard-RAxML/raxmlHPC-PTHREADS-AVX"
w <- sample(1:5, ncol(ips.cox1), replace = TRUE)

\dontrun{

# Simple tree search with GTRCAT and GTRGAMMA
tr <- raxml(ips.cox1, f = "d", N = 2, p = 1234,
           exec = exec) # -1743.528461
tr <- raxml(ips.cox1, m = "GTRGAMMA", f = "d", N = 2, p = 1234,
           exec = exec)
   
# Applying weights to columns                   
tr <- raxml(ips.cox1, f = "d", N = 2, p = 1234,
           weights = w, exec = exec) # -1743.528461

# Rapid bootstrap
tr <- raxml(ips.cox1, m = "GTRGAMMA",
           f = "a", N = 10, p = 1234, x = 1234,
           exec = exec)

# Rapid bootstrap with automatic halt
tr <- raxml(ips.cox1, m = "GTRGAMMA",
           f = "a", N = "autoMRE", p = 1234, x = 1234,
           exec = exec)
}
}
\references{
(in chronological order)

 Stamatakis, A., T. Ludwig and H. Meier. 2004. RAxML-III: A fast program for
 maximum likelihood-based inference of large phylogenetic trees.
 \emph{Bioinformatics} \bold{1}: 1--8.

 Stamatakis, A. 2006. RAxML-VI-HPC: Maximum likelihood-based phylogenetic
 analyses with thousands of taxa and mixed models. \emph{Bioinformatics}
 \bold{22}: 2688--2690.

 Stamatakis, A., P. Hoover, and J. Rougemont. 2008. A rapid bootstrap
 algorithm for the RAxML web-servers. \emph{Syst. Biol.} \bold{75}: 758--771.

 Pattengale, N. D., M. Alipour, O. R. P. Bininda-Emonds, B. M. E. Moret, and
 A. Stamatakis. 2010. How many bootstrap replicates are necessary?
 \emph{Journal of Computational Biology} \bold{17}: 337-354.

 Stamatakis, A. 2014. RAxML Version 8: A tool for phylogenetic analysis and
 post-analysis of large phylogenies. \emph{Bioinformatics} Advance Access.
}
\seealso{
\code{\link{raxml.partitions}} to store partitioning information in a
 data frame suitable for input as \code{partitions} argument in \code{raxml}.
}
