% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/07_interpolation.R
\name{interpolation}
\alias{interpolation}
\title{Interpolation for iAR, CiAR, and BiAR Classes}
\usage{
interpolation(x, ...)
}
\arguments{
\item{x}{An object of class \code{iAR}, \code{CiAR}, or \code{BiAR}, containing the model specification and parameters:
\itemize{
  \item For \code{iAR}:
    \itemize{
      \item \code{family}: The distribution family of the iAR model (one of "norm", "t", or "gamma").
      \item \code{series}: A numeric vector representing the time series to interpolate.
      \item \code{coef}: The coefficients of the iAR model.
      \item \code{zero_mean}: Logical, whether the model assumes a zero-mean series.
      \item \code{standardized}: Logical, whether the model uses standardized data (only for "norm" family).
      \item \code{mean}: The mean parameter (only for "gamma" family).
    }
  \item For \code{CiAR}:
    \itemize{
      \item \code{coef}: The coefficients of the CiAR model.
      \item \code{series_esd}: The series of error standard deviations for the CiAR model.
      \item \code{zero_mean}: Logical, whether the model assumes a zero-mean series.
      \item \code{standardized}: Logical, whether the model uses standardized data.
      \item \code{seed}: Optional seed for random number generation.
    }
  \item For \code{BiAR}:
    \itemize{
      \item \code{coef}: The coefficients of the BiAR model.
      \item \code{series_esd}: The series of error standard deviations for the BiAR model.
      \item \code{zero_mean}: Logical, whether the model assumes a zero-mean series.
      \item \code{yini1}: Initial value for the first time series (for BiAR models).
      \item \code{yini2}: Initial value for the second time series (for BiAR models).
      \item \code{seed}: Optional seed for random number generation.
    }
}}

\item{...}{Additional arguments (unused).}
}
\value{
An object of the same class as \code{x} with interpolated time series.
}
\description{
This method performs imputation of missing values in a time series using an autoregressive model.
The imputation is done iteratively for each missing value, utilizing available data and model coefficients.
Depending on the model family, the interpolation is performed differently:
- For \code{norm}: A standard autoregressive model for normally distributed data.
- For \code{t}: A model for time series with t-distributed errors.
- For \code{gamma}: A model for time series with gamma-distributed errors.
- For \code{CiAR}: A complex irregular autoregressive model.
- For \code{BiAR}: A bivariate autoregressive model.
}
\details{
Performs interpolation on time series with missing values. This method is implemented for:
1. Irregular Autoregressive models (iAR)
2. Complex Irregular Autoregressive models (CiAR)
3. Bivariate Autoregressive models (BiAR)


The method handles missing values (NA) in the time series by imputing them iteratively. 
For each missing value, the available data is used to fit the autoregressive model, and the missing value is imputed based on the model's output.
For the \code{CiAR} and \code{BiAR} models, the error standard deviations and initial values are also considered during imputation.
}
\examples{
# Interpolation for iAR model
library(iAR)
n=100
set.seed(6714)
o=iAR::utilities()
o<-gentime(o, n=n)
times=o@times
model_norm <- iAR(family = "norm", times = times, coef = 0.9)  
model_norm <- sim(model_norm)
y=model_norm@series
y1=y/sd(y)
model_norm@series=y1
model_norm@series_esd=rep(0,n)
model_norm <- kalman(model_norm) 
print(model_norm@coef)
napos=10
model_norm@series[napos]=NA
model_norm <- interpolation(model_norm)
interpolation=na.omit(model_norm@interpolated_values)
mse=as.numeric(y1[napos]-interpolation)^2
print(mse)
plot(times,y,type='l',xlim=c(times[napos-5],times[napos+5]))
points(times,y,pch=20)
points(times[napos],interpolation*sd(y),col="red",pch=20)

# Interpolation for CiAR model
model_CiAR <- CiAR(times = times,coef = c(0.9, 0))
model_CiAR <- sim(model_CiAR)
y=model_CiAR@series
y1=y/sd(y)
model_CiAR@series=y1
model_CiAR@series_esd=rep(0,n)
model_CiAR <- kalman(model_CiAR)
print(model_CiAR@coef)
napos=10
model_CiAR@series[napos]=NA
model_CiAR <- interpolation(model_CiAR)
interpolation=na.omit(model_CiAR@interpolated_values)
mse=as.numeric(y1[napos]-interpolation)^2
print(mse)
plot(times,y,type='l',xlim=c(times[napos-5],times[napos+5]))
points(times,y,pch=20)
points(times[napos],interpolation*sd(y),col="red",pch=20)
# Interpolation for BiAR model
model_BiAR <- BiAR(times = times,coef = c(0.9, 0.3), rho = 0.9)
model_BiAR <- sim(model_BiAR)
y=model_BiAR@series
y1=y/apply(y,2,sd)
model_BiAR@series=y1
model_BiAR@series_esd=matrix(0,n,2)
model_BiAR <- kalman(model_BiAR)
print(model_BiAR@coef) 
napos=10
model_BiAR@series[napos,1]=NA
model_BiAR <- interpolation(model_BiAR)
interpolation=na.omit(model_BiAR@interpolated_values[,1])
mse=as.numeric(y1[napos,1]-interpolation)^2
print(mse)
par(mfrow=c(2,1))
plot(times,y[,1],type='l',xlim=c(times[napos-5],times[napos+5]))
points(times,y[,1],pch=20)
points(times[napos],interpolation*apply(y,1,sd)[1],col="red",pch=20)
plot(times,y[,2],type='l',xlim=c(times[napos-5],times[napos+5]))
points(times,y[,2],pch=20)
}
\references{
\insertRef{Eyheramendy_2018}{iAR},\insertRef{Elorrieta_2019}{iAR},\insertRef{Elorrieta_2021}{iAR}
}
\seealso{
\code{\link{forecast}} for forecasting methods for these models.
}
