% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R
\name{mcmc_sample}
\alias{mcmc_sample}
\title{Hamiltonian Monte Carlo Sampler for Model Random Effects}
\usage{
mcmc_sample(
  Z,
  L,
  X,
  y,
  beta,
  family,
  link,
  warmup,
  nsamp,
  lambda,
  var_par = 1,
  trace = 0L,
  refresh = 500L,
  maxsteps = 100L,
  target_accept = 0.9
)
}
\arguments{
\item{Z}{Matrix Z of the GLMM}

\item{L}{Matrix L, the Cholesky decomposition of the random effect covariance (returned with `gen_chol_D()` function of covariance)}

\item{X}{Matrix X of the GLMM}

\item{y}{Vector of observations}

\item{beta}{Vector of parameters multiplying the X matrix}

\item{family}{Character specifying the family}

\item{link}{Character specifying the link function}

\item{warmup}{Integer. The number of warmup iterations for the MCMC sampler. Note, this reduces to 10 after
the first iteration as the sampler starts from the last set of values and adaptive step size}

\item{nsamp}{Integer. Total number of MCMC samples to draw on each iteration}

\item{lambda}{Value of the trajectory length of the leapfrog integrator in Hamiltonian Monte Carlo
(equal to number of steps times the step length). Larger values result in lower correlation in samples, but
require larger numbers of steps and so is slower.}

\item{var_par}{(Optional) Value of the scale parameter where required (Gaussian and gamma distributions)}

\item{trace}{Integer indicating what to report to the console, 0= nothing, 1-3=ever more detailed output}

\item{refresh}{Integer. Number of MCMC iterations to print progress to the console (requires verbose=TRUE)}

\item{maxsteps}{Integer. The maximum number of steps of the leapfrom integrator}

\item{target_accept}{The target acceptance rate of HMC proposals (default 0.9)}
}
\value{
A matrix (of dimension number of random effects * number of samples)
}
\description{
Hamiltonian Monte Carlo Sampler for Model Random Effects.
}
\examples{
## small example with simulated data
df <- nelder(~(j(10) * t(3)) > i(5))
des <- ModelMCML$new(
 covariance = list(
  formula =  ~(1|gr(j)*ar1(t)),
  parameters = c(0.25,0.7)
),
mean = list(
  formula = ~factor(t)-1,
  parameters = rnorm(3)
),
data=df,
family=gaussian()
)
## simulate data
y <- des$sim_data()
## get covariance definition matrix
ddata <- des$covariance$get_D_data()
## simulate some values of the random effects
## first, we need to extract the Cholesky decomposition of the covariance matrix D
L <- des$covariance$get_chol_D()
## generate samples using HMC
mat <- mcmc_sample(Z = as.matrix(des$covariance$Z),
   L = as.matrix(L),
   X = as.matrix(des$mean_function$X),
   y = as.vector(y),
   beta = des$mean_function$parameters,
   var_par = 1,
   family = des$mean_function$family[[1]],
   link = des$mean_function$family[[2]],
   warmup = 250, 
   nsamp = 250,
   lambda = 5,
   maxsteps = 100,
   trace=1,
   target_accept = 0.95)
}
