% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/stat-quadrant-counts.R
\name{stat_quadrant_counts}
\alias{stat_quadrant_counts}
\title{Number of observations in quadrants}
\usage{
stat_quadrant_counts(
  mapping = NULL,
  data = NULL,
  geom = "text_npc",
  position = "identity",
  quadrants = NULL,
  pool.along = "none",
  xintercept = 0,
  yintercept = 0,
  label.x = NULL,
  label.y = NULL,
  na.rm = FALSE,
  show.legend = FALSE,
  inherit.aes = TRUE,
  ...
)
}
\arguments{
\item{mapping}{The aesthetic mapping, usually constructed with
\code{\link[ggplot2]{aes}} or \code{\link[ggplot2]{aes_}}. Only needs to be
set at the layer level if you are overriding the plot defaults.}

\item{data}{A layer specific dataset - only needed if you want to override
the plot defaults.}

\item{geom}{The geometric object to use display the data}

\item{position}{The position adjustment to use for overlapping points on this
layer}

\item{quadrants}{integer vector indicating which quadrants are of interest,
with a \code{OL} indicating the whole plot.}

\item{pool.along}{character, one of "none", "x" or "y", indicating which
quadrants to pool to calculate counts by pair of quadrants.}

\item{xintercept, yintercept}{numeric the coordinates of the origin of the
quadrants.}

\item{label.x, label.y}{\code{numeric} Coordinates (in npc units) to be used
for absolute positioning of the labels.}

\item{na.rm}{a logical indicating whether NA values should be stripped before
the computation proceeds.}

\item{show.legend}{logical. Should this layer be included in the legends?
\code{NA}, the default, includes if any aesthetics are mapped. \code{FALSE}
never includes, and \code{TRUE} always includes.}

\item{inherit.aes}{If \code{FALSE}, overrides the default aesthetics, rather
than combining with them. This is most useful for helper functions that
define both data and aesthetics and should not inherit behaviour from the
default plot specification, e.g. \code{\link[ggplot2]{borders}}.}

\item{...}{other arguments passed on to \code{\link[ggplot2]{layer}}. This
can include aesthetics whose values you want to set, not map. See
\code{\link[ggplot2]{layer}} for more details.}
}
\description{
\code{stat_quadrant_counts()} counts the number of observations in each
quadrant of a plot panel. By default it adds a text label to the far corner
of each quadrant. It can also be used to obtain the total number of
observations in each of two pairs of quadrants or in the whole panel.
Grouping is ignored, so en every case a single count is computed for each
quadrant in a plot panel.
}
\details{
This stat can be used to automatically count observations in each of
  the four quadrants of a plot, and by default add these counts as text
  labels. Values exactly equal to zero are counted as belonging to the
  positve quadrant. An argument value of zero, passed to formal parameter
  \code{quadrants} is interpreted as a request for the count of all
  observations in each plot panel.

  The default origin of quadrants is at \code{xintercept = 0},
  \code{yintercept = 0}. Alsoby default, counts are computed for all
  quadrants within the $x$ and $y$ scale limits, but ignoring any marginal
  scale expansion. The default positions of the labels is in the farthest
  corner or edge of each quadrant using npc coordinates. Consequently, when
  using facets even with free limits for $x$ and $y$ axes, the location of
  the labels is consistent across panels. This is achieved by use of
  \code{geom = "text_npc"} or \code{geom = "label_npc"}. To pass the
  positions in native data units, pass \code{geom = "text"} explicitly as
  argument.
}
\section{Computed variables}{
 Data frame with one to four rows, one for each
  quadrant for which counts are counted in \code{data}. \describe{
  \item{quadrant}{integer, one of 0:4} \item{x}{x value of label position in
  data units} \item{y}{y value of label position in data units} \item{npcx}{x
  value of label position in npc units} \item{npcy}{y value of label position
  in npc units} \item{count}{number of  observations} }.

  As shown in one example below \code{\link[gginnards]{geom_debug}} can be
  used to print the computed values returned by any statistic. The output
  shown includes also values mapped to aesthetics, like \code{label} in the
  example.
}

\examples{
library(gginnards)
# generate artificial data
set.seed(4321)
x <- 1:100
y <- rnorm(length(x), mean = 10)
my.data <- data.frame(x, y)

ggplot(my.data, aes(x, y)) +
  geom_point() +
  stat_quadrant_counts()

# We use geom_debug() to see the computed values
ggplot(my.data, aes(x, y)) +
  geom_point() +
  stat_quadrant_counts(geom = "debug")

ggplot(my.data, aes(x, y)) +
  geom_point() +
  stat_quadrant_counts(aes(label = sprintf("\%i observations", stat(count)))) +
  expand_limits(y = 12.7)

ggplot(my.data, aes(x, y)) +
  geom_quadrant_lines(colour = "blue", xintercept = 50, yintercept = 10) +
  stat_quadrant_counts(colour = "blue", xintercept = 50, yintercept = 10) +
  geom_point() +
  scale_y_continuous(expand = expansion(mult = 0.15, add = 0))

ggplot(my.data, aes(x, y)) +
  geom_quadrant_lines(colour = "blue",
                       pool.along = "x", yintercept = 10) +
  stat_quadrant_counts(colour = "blue", label.x = "right",
                       pool.along = "x", yintercept = 10) +
  geom_point() +
  expand_limits(y = c(7, 13))

ggplot(my.data, aes(x, y)) +
  geom_point() +
  stat_quadrant_counts(quadrants = 0, label.x = "left", label.y = "bottom")

ggplot(my.data, aes(x, y)) +
  geom_point() +
  stat_quadrant_counts(geom = "text") # use "tex" instead

}
\seealso{
Other Functions for quadrant and volcano plots: 
\code{\link{FC_format}()},
\code{\link{geom_quadrant_lines}()},
\code{\link{outcome2factor}()},
\code{\link{scale_colour_outcome}()},
\code{\link{scale_shape_outcome}()},
\code{\link{scale_y_Pvalue}()},
\code{\link{xy_outcomes2factor}()}
}
\concept{Functions for quadrant and volcano plots}
