% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/weighted-statistics.R
\name{w_mean}
\alias{w_mean}
\alias{w_var}
\alias{w_sd}
\alias{w_cov}
\alias{w_cor}
\alias{w_quantile}
\alias{w_median}
\alias{w_std}
\alias{w_scale}
\alias{w_center}
\title{Calculate weighted statistics}
\usage{
w_mean(x, w = NULL, na.rm = FALSE)

w_var(x, w = NULL, na.rm = FALSE)

w_sd(x, w = NULL, na.rm = FALSE)

w_cov(x, w = NULL, na.rm = FALSE)

w_cor(x, w = NULL)

w_quantile(
  x,
  w = NULL,
  probs = seq(0, 1, by = 0.25),
  na.rm = FALSE,
  names = TRUE,
  type = 7L,
  digits = 7L
)

w_median(x, w = NULL, na.rm = FALSE)

w_std(x, w = NULL, na.rm = TRUE, scale = TRUE, center = TRUE)

w_scale(x, w = NULL, na.rm = TRUE)

w_center(x, w = NULL, na.rm = TRUE)
}
\arguments{
\item{x}{a numeric variable; for \code{w_cov()} and \code{w_cor()}, a numeric matrix.}

\item{w}{optional; a numeric vector of weights with length equal to the length of \code{x} (or number of rows if \code{x} is a matrix). If unspecified, will use bootstrapped weights when called from within \code{\link[=fwb]{fwb()}} or \code{\link[=vcovFWB]{vcovFWB()}} and unit weights (i.e., for unweighted estimates) otherwise.}

\item{na.rm}{logical; whether to exclude \code{NA} values in the weights and \code{x} when computing statistics. Default is \code{FALSE} for the weighted statistics (like with their unweighted counterparts) and \code{TRUE} for weighted transformations.}

\item{probs, names, type, digits}{see \code{\link[=quantile]{quantile()}}. Only \code{type = 7} is allowed.}

\item{scale, center}{logical; whether to scale or center the variable.}
}
\value{
\code{w_mean()}, \code{w_var()}, \code{w_sd()}, and \code{w_median()} return a numeric scalar. \code{w_cov()} and \code{w_cor()} return numeric matrices. \code{w_quantile()} returns a numeric vector of length equal to \code{probs}. \code{w_std()}, \code{w_scale()}, and \code{w_center()} return numeric vectors of length equal to the length of \code{x}.
}
\description{
These functions are designed to compute weighted statistics (mean, variance, standard deviation, covariance, correlation, median and quantiles) to perform weighted transformation (scaling, centering, and standardization) using bootstrap weights. These automatically extract bootstrap weights when called from within \code{\link[=fwb]{fwb()}} to facilitate computation of bootstrap statistics without needing to think to hard about how to correctly incorporate weights.
}
\details{
These function automatically incorporate bootstrap weights when used inside \code{\link[=fwb]{fwb()}} or \code{\link[=vcovFWB]{vcovFWB()}}. This works because \code{fwb()} and \code{vcovFWB()} temporarily set an option with the environment of the function that calls the estimation function with the sampled weights, and the \verb{w_*()} functions access the bootstrap weights from that environment, if any. So, using, e.g., \code{w_mean()} inside the function supplied to the \code{statistic} argument of \code{fwb()}, computes the weighted mean using the bootstrap weights. Using these functions outside \code{fwb()} works just like other functions that compute weighted statistics: when \code{w} is supplied, the statistics are weighted, and otherwise they are unweighted.

See below for how each statistic is computed.
\subsection{Weighted statistics}{

For all weighted statistics, the weights are first rescaled to sum to 1. \eqn{w} in the formulas below refers to these weights.

\describe{
\item{\code{w_mean()}}{Calculates the weighted mean as
\deqn{\bar{x}_w = \sum_i{w_i x_i}}
This is the same as \code{\link[=weighted.mean]{weighted.mean()}}.}
\item{\code{w_var()}}{Calculates the weighted variance as
\deqn{s^2_{x,w} = \frac{\sum_i{w_i(x_i - \bar{x}_w)^2}}{1 - \sum_i{w_i^2}}}}
\item{\code{w_sd()}}{Calculates the weighted standard deviation as
\deqn{s_{x,w} = \sqrt{s^2_{x,w}}}}
\item{\code{w_cov()}}{Calculates the weighted covariances as
\deqn{s_{x,y,w} = \frac{\sum_i{w_i(x_i - \bar{x}_w)(y_i - \bar{y}_w)}}{1 - \sum_i{w_i^2}}}
This is the same as \code{\link[=cov.wt]{cov.wt()}}.}
\item{\code{w_cor()}}{Calculates the weighted correlation as
\deqn{r_{x,y,w} = \frac{s_{x,y,w}}{s_{x,w}s_{y,w}}}
This is the same as \code{\link[=cov.wt]{cov.wt()}}.}
\item{\code{w_quantile()}}{Calculates the weighted quantiles using linear interpolation of the weighted cumulative density function.}
\item{\code{w_median()}}{Calculates the weighted median as \code{w_quantile(., probs = .5)}.}
}
}

\subsection{Weighted transformations}{

Weighted transformations use the weighted mean and/or standard deviation to re-center or re-scale the given variable. In the formulas below, \eqn{x} is the original variable and \eqn{w} are the weights.

\describe{
\item{\code{w_center()}}{Centers the variable at its (weighted) mean.
\deqn{x_{c,w} = x - \bar{x}_w}}
\item{\code{w_scale()}}{Scales the variable by its (weighted) standard deviation.
\deqn{x_{s,w} = x / s_{x,w}}}
\item{\code{w_std()}}{Centers and scales the variable by its (weighted) mean and standard deviation.
\deqn{x_{z,w} = (x - \bar{x}_w) / s_{x,w}}}
}

\code{w_scale()} and \code{w_center()} are efficient wrappers to \code{w_std()} with \code{center = FALSE} and \code{scale = FALSE}, respectively.
}
}
\examples{
\dontshow{if (requireNamespace("cobalt", quietly = TRUE) && requireNamespace("lmtest", quietly = TRUE)) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
# G-computation of average treatment effects using lalonde
# dataset

data("lalonde", package = "cobalt")

ate_est <- function(data, w) {
  fit <- lm(re78 ~ treat * (age + educ + married + race +
                              nodegree + re74 + re75),
            data = data, weights = w)

  p0 <- predict(fit, newdata = transform(data, treat = 0))
  p1 <- predict(fit, newdata = transform(data, treat = 1))

  # Weighted means using bootstrap weights
  m0 <- w_mean(p0)
  m1 <- w_mean(p1)

  c(m0 = m0, m1 = m1, ATE = m1 - m0)
}

set.seed(123, "L'Ecuyer-CMRG")
boot_est <- fwb(lalonde, statistic = ate_est,
                R = 199, verbose = FALSE)
summary(boot_est)

# Using `w_*()` data transformations inside a model
# supplied to vcovFWB():
fit <- lm(re78 ~ treat * w_center(age),
          data = lalonde)

lmtest::coeftest(fit, vcov = vcovFWB, R = 500)
\dontshow{\}) # examplesIf}
}
\seealso{
\itemize{
\item \code{\link[=mean]{mean()}} and \code{\link[=weighted.mean]{weighted.mean()}} for computing the unweighted and weighted mean
\item \code{\link[=var]{var()}} and \code{\link[=sd]{sd()}} for computing the unweighted variance and standard deviation
\item \code{\link[=median]{median()}} and \code{\link[=quantile]{quantile()}} for computing the unweighted median and quantiles
\item \code{\link[=cov]{cov()}}, \code{\link[=cor]{cor()}}, and \code{\link[=cov.wt]{cov.wt()}} for unweighted and weighted covariance and correlation matrices
\item \code{\link[=scale]{scale()}} for standardizing variables using arbitrary (by default, unweighted) centering and scaling factors
}
}
