% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/combine_forecasts.R
\name{combine_forecasts}
\alias{combine_forecasts}
\title{Combine multiple horizon-specific forecast models to produce one forecast}
\usage{
combine_forecasts(
  ...,
  type = c("horizon", "error"),
  data_error = list(NULL),
  metric = NULL
)
}
\arguments{
\item{...}{One or more objects of class 'forecast_results' from running \code{predict.forecast_model()} on
an input forward-looking forecast dataset. These are the forecasts from the horizon-specific
direct forecasting models trained over the entire training dataset by setting \code{create_windows(..., window_length = 0)}.
If multiple models are passed in \code{...}, the model names from \code{train_model()} should be unique for a
given model forecast horizon.}

\item{type}{Default: 'horizon'. A character vector of length 1 that identifies the forecast combination method.}

\item{data_error}{Optional. A list of objects of class 'validation_error' from running \code{return_error()}
on a training dataset. The length and order of \code{data_error} should match the models passed in \code{...}.}

\item{metric}{Required if \code{data_error} is given. A length 1 character vector naming the forecast
error metric used to select the optimal model at each forecast horizon from the models passed
in '...' e.g., 'mae'.}
}
\value{
An S3 object of class 'forecastML' with final h-step-ahead forecasts.

   \strong{Forecast combination type:}
    \itemize{
      \item \code{type = 'horizon'}: 1 final h-step-ahead forecast is returned for each model object passed in \code{...}.
      \item \code{type = 'error'}: 1 final h-step-ahead forecast is returned by selecting, for each forecast horizon,
      the model that minimized the chosen error metric at that horizon on the outer-loop validation data sets.
   }

   \strong{Columns in returned 'forecastML' data.frame:}
    \itemize{
      \item \code{model}: User-supplied model name in \code{train_model()}.
      \item \code{model_forecast_horizon}: The direct-forecasting time horizon that the model was trained on.
      \item \code{horizon}: Forecast horizons, 1:h, measured in dataset rows.
      \item \code{forecast_period}: The forecast period in row indices or dates. The forecast period starts at either \code{attributes(create_lagged_df())$data_stop + 1} for row indices or \code{attributes(create_lagged_df())$data_stop + 1 * frequency} for date indices.
      \item \code{"groups"}: If given, the user-supplied groups in \code{create_lagged_df()}.
      \item \code{"outcome_name"_pred}: The final forecasts.
      \item \code{"outcome_name"_pred_lower}: If given, the lower forecast bounds returned by the user-supplied prediction function.
      \item \code{"outcome_name"_pred_upper}: If given, the upper forecast bounds returned by the user-supplied prediction function.
   }
}
\description{
The horizon-specific models can either be combined to (a) produce final forecasts for only those
horizons at which they were trained (i.e., shorter-horizon models override longer-horizon models
when producing final short-horizon h-step-ahead forecasts) or (b) produce final forecasts using
any combination of horizon-specific models that minimized error over the validation/training dataset.
}
\section{Methods and related functions}{


The output of \code{combine_forecasts()} has the following generic S3 methods

\itemize{
  \item \code{\link[=plot.forecastML]{plot}}
}
}

\examples{
# Example with "type = 'horizon'".
data("data_seatbelts", package = "forecastML")

horizons <- c(1, 3, 12)
lookback <- 1:15

data_train <- create_lagged_df(data_seatbelts, type = "train", outcome_col = 1,
                               lookback = lookback, horizon = horizons)

windows <- create_windows(data_train, window_length = 0)

model_function <- function(data, my_outcome_col) {
  model <- lm(DriversKilled ~ ., data = data)
  return(model)
}

model_results <- train_model(data_train, windows, model_name = "LM", model_function)

data_forecast <- create_lagged_df(data_seatbelts, type = "forecast", outcome_col = 1,
                                  lookback = lookback, horizon = horizons)

prediction_function <- function(model, data_features) {
  x <- data_features
  data_pred <- data.frame("y_pred" = predict(model, newdata = x))
  return(data_pred)
}

data_forecasts <- predict(model_results, prediction_function = list(prediction_function),
                          data = data_forecast)

data_combined <- combine_forecasts(data_forecasts)

plot(data_combined)
}
