% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/eim-class.R
\name{bootstrap}
\alias{bootstrap}
\alias{bootstrap()}
\title{Runs a Bootstrap to Estimate the \strong{Standard Deviation} of Predicted Probabilities}
\usage{
bootstrap(
  object = NULL,
  X = NULL,
  W = NULL,
  json_path = NULL,
  nboot = 50,
  allow_mismatch = TRUE,
  seed = NULL,
  ...
)
}
\arguments{
\item{object}{An object of class \code{eim}, which can be created using the \link{eim} function. This parameter should not be used if either (i) \code{X} and \code{W} matrices or (ii) \code{json_path} is supplied. See \strong{Note}.}

\item{X}{A \verb{(b x c)} matrix representing candidate votes per ballot box.}

\item{W}{A \verb{(b x g)} matrix representing group votes per ballot box.}

\item{json_path}{A path to a JSON file containing \code{X} and \code{W} fields, stored as nested arrays. It may contain additional fields with other attributes, which will be added to the returned object.}

\item{nboot}{Integer specifying how many times to run the
EM algorithm.}

\item{allow_mismatch}{Boolean, if \code{TRUE}, allows a mismatch between the voters and votes for each ballot-box, only works if \code{method} is \code{"mvn_cdf"}, \code{"mvn_pdf"}, \code{"mult"} and \code{"mcmc"}. If \code{FALSE}, throws an error if there is a mismatch. By default it is \code{TRUE}.}

\item{seed}{An optional integer indicating the random seed for the randomized algorithms. This argument is only applicable if \code{initial_prob = "random"} or \code{method} is either \code{"mcmc"} or \code{"mvn_cdf"}. Aditionally, it sets the random draws of the ballot boxes.}

\item{...}{Additional arguments passed to the \link{run_em} function that will execute the EM algorithm.}
}
\value{
Returns an \code{eim} object with the \code{sd} field containing the estimated standard deviations of the probabilities and the amount of iterations that were made. If an \code{eim} object is provided, its attributes (see \link{run_em}) are retained in the returned object.
}
\description{
This function computes the Expected-Maximization (EM) algorithm "\code{nboot}" times. It then computes the standard deviation from the \code{nboot} estimated probability matrices on each component.
}
\note{
This function can be executed using one of three mutually exclusive approaches:
\enumerate{
\item By providing an existing \code{eim} object.
\item By supplying both input matrices (\code{X} and \code{W}) directly.
\item By specifying a JSON file (\code{json_path}) containing the matrices.
}

These input methods are \strong{mutually exclusive}, meaning that you must provide \strong{exactly one} of
these options. Attempting to provide more than one or none of these inputs will result in an error.

When called with an \code{eim} object, the function updates the object with the computed results.
If an \code{eim} object is not provided, the function will create one internally using either the
supplied matrices or the data from the JSON file before executing the algorithm.
}
\examples{
\donttest{
# Example 1: Using an 'eim' object directly
simulations <- simulate_election(
    num_ballots = 200,
    num_candidates = 5,
    num_groups = 3,
)

model <- eim(X = simulations$X, W = simulations$W)

model <- bootstrap(
    object = model,
    nboot = 30,
    method = "mult",
    maxiter = 500,
    verbose = FALSE,
)

# Access standard deviation throughout 'model'
print(model$sd)


# Example 2: Providing 'X' and 'W' matrices directly
model <- bootstrap(
    X = simulations$X,
    W = simulations$W,
    nboot = 15,
    method = "mvn_pdf",
    maxiter = 100,
    maxtime = 5,
    param_threshold = 0.01,
    allow_mismatch = FALSE
)

print(model$sd)
}

# Example 3: Using a JSON file as input

\dontrun{
model <- bootstrap(
    json_path = "path/to/election_data.json",
    nboot = 70,
    method = "mult",
)

print(model$sd)
}

}
\seealso{
The \link{eim} object and \link{run_em} implementation.
}
