#' @title Create the underlying node and edge data frames
#'   behind [vis_drake_graph()].
#' @description With the returned data frames,
#' you can plot your own custom `visNetwork` graph.
#' @export
#' @return A list of three data frames: one for nodes,
#'   one for edges, and one for
#'   the legend nodes. The list also contains the
#'   default title of the graph.
#' @seealso [vis_drake_graph()]
#' @param config A [drake_config()] configuration list.
#'   You can get one as a return value from [make()] as well.
#'
#' @param from Optional collection of target/import names.
#'   If `from` is nonempty,
#'   the graph will restrict itself to
#'   a neighborhood of `from`.
#'   Control the neighborhood with
#'   `mode` and `order`.
#'
#' @param mode Which direction to branch out in the graph
#'   to create a neighborhood around `from`.
#'   Use `"in"` to go upstream,
#'   `"out"` to go downstream,
#'   and `"all"` to go both ways and disregard
#'   edge direction altogether.
#'
#' @param order How far to branch out to create
#'   a neighborhood around `from`. Defaults to
#'   as far as possible. If a target is in the neighborhood, then
#'   so are all of its custom [file_out()] files if
#'   `show_output_files` is `TRUE`.
#'   That means the actual graph order may be slightly greater than
#'   you might expect, but this ensures consistency
#'   between `show_output_files = TRUE` and
#'   `show_output_files = FALSE`.
#'
#' @param subset Optional character vector.
#'   Subset of targets/imports to display in the graph.
#'   Applied after `from`, `mode`, and `order`.
#'   Be advised: edges are only kept for adjacent nodes in `subset`.
#'   If you do not select all the intermediate nodes,
#'   edges will drop from the graph.
#'
#' @param targets_only Logical,
#'   whether to skip the imports and only include the
#'   targets in the workflow plan.
#'
#' @param font_size Numeric, font size of the node labels in the graph
#'
#' @param build_times Character string or logical.
#'   If character, the choices are
#'     1. `"build"`: runtime of the command plus the time
#'       it take to store the target or import.
#'     2. `"command"`: just the runtime of the command.
#'     3. `"none"`: no build times.
#'   If logical, `build_times` selects whether to show the
#'   times from `build_times(..., type = "build")`` or use
#'   no build times at all. See [build_times()] for details.
#'
#' @param digits Number of digits for rounding the build times
#'
#' @param from_scratch Logical, whether to assume all the targets
#'   will be made from scratch on the next [make()].
#'   Makes all targets outdated, but keeps information about
#'   build progress in previous [make()]s.
#'
#' @param make_imports Logical, whether to make the imports first.
#'   Set to `FALSE` to increase speed and risk using obsolete information.
#'
#' @param full_legend Logical. If `TRUE`, all the node types
#'   are printed in the legend. If `FALSE`, only the
#'   node types used are printed in the legend.
#'
#' @param group Optional character scalar, name of the column used to
#'   group nodes into columns. All the columns names of your original `drake`
#'   plan are choices. The other choices (such as `"status"`) are column names
#'   in the `nodes` . To group nodes into clusters in the graph,
#'   you must also supply the `clusters` argument.
#'
#' @param clusters Optional character vector of values to cluster on.
#'   These values must be elements of the column of the `nodes` data frame
#'   that you specify in the `group` argument to `drake_graph_info()`.
#'
#' @param show_output_files Logical, whether to include
#'   [file_out()] files in the graph.
#'
#' @param hover Logical, whether to show text (file contents,
#'   commands, etc.) when you hover your cursor over a node.
#'
#' @examples
#' \dontrun{
#' isolate_example("Quarantine side effects.", {
#' if (requireNamespace("visNetwork", quietly = TRUE)) {
#' if (suppressWarnings(require("knitr"))) {
#' load_mtcars_example() # Get the code with drake_example("mtcars").
#' config <- drake_config(my_plan) # my_plan loaded with load_mtcars_example()
#' vis_drake_graph(config) # Jump straight to the interactive graph.
#' # Get a list of data frames representing the nodes, edges,
#' # and legend nodes of the visNetwork graph from vis_drake_graph().
#' raw_graph <- drake_graph_info(config = config)
#' # Choose a subset of the graph.
#' smaller_raw_graph <- drake_graph_info(
#'   config = config,
#'   from = c("small", "reg2"),
#'   mode = "in"
#' )
#' # Inspect the raw graph.
#' str(raw_graph)
#' # Use the data frames to plot your own custom visNetwork graph.
#' # For example, you can omit the legend nodes
#' # and change the direction of the graph.
#' library(visNetwork)
#' graph <- visNetwork(nodes = raw_graph$nodes, edges = raw_graph$edges)
#' visHierarchicalLayout(graph, direction = 'UD')
#' }
#' }
#' })
#' }
drake_graph_info <- function(
  config,
  from = NULL,
  mode = c("out", "in", "all"),
  order = NULL,
  subset = NULL,
  build_times = "build",
  digits = 3,
  targets_only = FALSE,
  font_size = 20,
  from_scratch = FALSE,
  make_imports = TRUE,
  full_legend = FALSE,
  group = NULL,
  clusters = NULL,
  show_output_files = TRUE,
  hover = FALSE
) {
  assert_pkg("visNetwork")
  assert_config_not_plan(config)
  if (!length(V(config$graph)$name)) {
    return(null_graph())
  }
  config$build_times <- resolve_build_times(build_times)
  config$digits <- digits
  config$font_size <- font_size
  config$from_scratch <- from_scratch
  config$make_imports <- make_imports
  config$group <- as.character(group)
  config$clusters <- as.character(clusters)
  config$hover <- hover
  config$file_out <- lapply(all_targets(config), function(target) {
    config$layout[[target]]$deps_build$file_out
  })
  names(config$file_out) <- all_targets(config)
  if (!show_output_files) {
    vertices <- igraph::V(config$graph)$name
    imported <- igraph::V(config$graph)$imported
    vertices <- vertices[!(!imported & is_encoded_path(vertices))]
    config$graph <- subset_graph(config$graph, vertices)
  }
  if (!is.null(from)) {
    config$graph <- nbhd_graph(
      graph = config$graph,
      vertices = from,
      mode = match.arg(mode),
      order = order %||% igraph::gorder(config$graph)
    )
  }
  if (!is.null(subset)) {
    config$graph <- subset_graph(graph = config$graph, subset = subset)
  }
  config$import_names <- all_imports(config)
  if (targets_only) {
    config$graph <- igraph::delete_vertices(
      graph = config$graph,
      v = igraph::V(config$graph)$name[igraph::V(config$graph)$imported]
    )
  }
  config <- get_raw_node_category_data(config)
  network_data <- visNetwork::toVisNetworkData(config$graph)
  network_data$nodes
  config$nodes <- network_data$nodes[igraph::topo_sort(config$graph)$name, ]
  if (!is.null(group)) {
    config$nodes[[group]] <- get_cluster_grouping(config, group)
  }
  config <- trim_node_categories(config)
  config$nodes <- configure_nodes(config = config)
  if (show_output_files) {
    config$nodes <- append_output_file_nodes(config)
  }
  config$nodes <- coord_set(config$nodes)
  config$edges <- network_data$edges
  if (nrow(config$edges)) {
    config$edges$arrows <- "to"
  }
  if (length(config$group)) {
    config <- cluster_nodes(config)
  }
  list(
    nodes = weak_as_tibble(config$nodes),
    edges = weak_as_tibble(config$edges),
    legend_nodes = filtered_legend_nodes(
      all_nodes = config$nodes,
      full_legend = full_legend,
      font_size = font_size
    ),
    default_title = default_graph_title()
  )
}
