% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/clusterabilitytest.R
\name{clusterabilitytest}
\alias{clusterabilitytest}
\title{Perform a test of clusterability}
\usage{
clusterabilitytest(data, test, reduction = "pca",
  distance_metric = "euclidean", distance_standardize = "std",
  pca_center = TRUE, pca_scale = TRUE, is_dist_matrix = FALSE,
  completecase = FALSE, d_simulatepvalue = FALSE, d_reps = 2000,
  s_m = 999, s_adjust = TRUE, s_digits = 6, s_setseed = NULL,
  s_outseed = FALSE)
}
\arguments{
\item{data}{the data set to be used in the test. Must contain only numeric data.}

\item{test}{the test to be performed. Either \code{"dip"} or \code{"silverman"}. See 'Details' section below for how to pick a test.}

\item{reduction}{any dimension reduction that is to be performed.
\itemize{
\item{\code{"none"} performs no dimension reduction.}
\item{\code{"pca"} uses the scores from the first principal component.}
\item{\code{"distance"} computes pairwise distances (using \code{distance_metric} as the metric).}
}
For multivariate \code{data}, dimension reduction is required.}

\item{distance_metric}{if applicable, the metric to be used in computing pairwise distances.

The \code{"euclidean"} (default), \code{"maximum", "manhattan", "canberra", "binary"} choices work the same as in \code{\link{dist}}. The Minkowski metric is available by providing \code{"minkowski(p)"}.

Additional choices are:
\itemize{
\item{\code{"sqeuc"}: squared Euclidean distances.}
\item{\code{"cov"}: covariance similarity coefficient,}
\item{\code{"corr"}: correlation similarity coefficient}
\item{\code{"sqcorr"}: squared correlation similarity coefficient.}
}

CAUTION: Not all of these have been tested, but instead are provided to potentially be useful. If in doubt, use the default \code{"euclidean"}.}

\item{distance_standardize}{how the variables should be standardized, if at all.
\itemize{
\item{\code{"none"}: no standardization is performed}
\item{\code{"std"} (default) each variable standardized to have mean 0 and standard deviation 1}
\item{\code{"mean"}: each variable standardized to have mean 0 (standard deviation is unchanged)}
\item{\code{"median"}: each variable standardized to have median 0 (standard deviation is unchanged)}
}}

\item{pca_center}{if applicable, a logical value indicating whether the variables should be shifted to be zero centered (see \code{\link{prcomp}} for more details). Default is \code{TRUE}.}

\item{pca_scale}{if applicable, a logical value indicating whether the variables should be scaled to have unit variance before the analysis takes place (see \code{\link{prcomp}} for details). Default is \code{TRUE}.}

\item{is_dist_matrix}{a logical value indicating whether the \code{data} argument is a distance matrix. If \code{TRUE} then the lower triangular portion of \code{data} will be extracted and be used in the multimodality test.}

\item{completecase}{a logical value indicating whether a complete case analysis should be performed. For both tests, missing data must be removed before the test can be performed. This can be done manually by the user or by setting \code{completecase = TRUE}.}

\item{d_simulatepvalue}{for Dip Test, a logical value indicating whether \eqn{p}~values should be obtained via Monte Carlo simulation (see \code{\link{dip.test}} for details).}

\item{d_reps}{for Dip Test, a positive integer. The number of replicates used in Monte Carlo simulation. Only used if \code{d_simulatepvalue} is \code{TRUE}.}

\item{s_m}{for Silverman Test, a positive integer. The number of bootstrap replicates used in the test. Default is \code{999}.}

\item{s_adjust}{for Silverman Test, a logical value indicating whether p-values are adjusted using work by @cite{hallyorksilverman}.}

\item{s_digits}{for Silverman Test, a positive integer indicating the number of digits to round the p value. Default is \code{6} and is only used when \code{s_adjust = TRUE}.}

\item{s_setseed}{for Silverman Test, an integer used to set the seed of the random number generator. If the default value of \code{NULL} is used, then no seed will be set.}

\item{s_outseed}{for Silverman Test, a logical value indicating whether to return the state of the random number generator as part of the output. This is used in limited cases for troubleshooting, so the default is \code{FALSE}.}
}
\value{
\code{clusterabilitytest} returns a \code{clusterability} object containing information on the test performed and results. Can be printed using the \code{\link{print.clusterability}} function.
}
\description{
Performs tests for clusterability of a data set and returns results in a clusterability object. Can do data reduction via PCA or pairwise distances and standardize data prior to performing the test.
}
\examples{
### Quick start ###
# Load data and remove Species
data(iris)
iris_num <- iris[,-5]
plot(iris_num)

# Run test using default options
clust_result <- clusterabilitytest(iris_num, "dip")

# Print results
print(clust_result)

\donttest{
### Longer Example: Specifying Parameters ###
# Load data and plot to visualize
data(normals2)
plot(normals2)

# Using Silverman's test, pairwise distances to reduce dimension,
# 1,000 bootstrap replicates, with an RNG seed of 12345
clust_result2 <- clusterabilitytest(normals2, "silverman", reduction = "distance",
     s_m = 1000, s_setseed = 12345)

# Print result
print(clust_result2)}

}
\references{
Hall, P. and York, M., 2001. On the calibration of Silverman's test for multimodality. Statistica Sinica, pp.515-536.

Silverman, B.W., 1981. Using kernel density estimates to investigate multimodality. Journal of the Royal Statistical Society. Series B (Methodological), pp.97-99.

Martin Maechler (2016). diptest: Hartigan's Dip Test Statistic for Unimodality - Corrected. R package version 0.75-7. https://CRAN.R-project.org/package=diptest

Schwaiger F, Holzmann H. Package which implements the silvermantest; 2013. Available from: https://www.mathematik.uni-marburg.de/stochastik/R packages/.
}
\seealso{
\code{\link{print.clusterability}}
}
