
#' @title Bayesian estimation of a Structural Vector Autoregression with 
#' Markov-switching heteroskedasticity via Gibbs sampler
#'
#' @description Estimates the SVAR with Markov-switching heteroskedasticity with \code{M} regimes (MS(M)) proposed by Woźniak & Droumaguet (2022).
#' Implements the Gibbs sampler proposed by Waggoner & Zha (2003)
#' for the structural matrix \eqn{B} and the equation-by-equation sampler by Chan, Koop, & Yu (2021)
#' for the autoregressive slope parameters \eqn{A}. Additionally, the parameter matrices \eqn{A} and \eqn{B}
#' follow a Minnesota prior and generalised-normal prior distributions respectively with the matrix-specific
#' overall shrinkage parameters estimated thanks to a 3-level hierarchical prior distribution. The MS
#' model is estimated using the prior distributions and algorithms proposed by Woźniak & Droumaguet (2022).
#' See section \bold{Details} for the model equations.
#' 
#' @details 
#' The heteroskedastic SVAR model is given by the reduced form equation:
#' \deqn{Y = AX + E}
#' where \eqn{Y} is an \code{NxT} matrix of dependent variables, \eqn{X} is a \code{KxT} matrix of explanatory variables, 
#' \eqn{E} is an \code{NxT} matrix of reduced form error terms, and \eqn{A} is an \code{NxK} matrix of autoregressive slope coefficients and parameters on deterministic terms in \code{X}.
#' 
#' The structural equation is given by
#' \deqn{BE = U}
#' where \eqn{U} is an \code{NxT} matrix of structural form error terms, and
#' \eqn{B} is an \code{NxN} matrix of contemporaneous relationships.
#' 
#' Finally, the structural shocks, \eqn{U}, are temporally and contemporaneously independent and jointly normally distributed with zero mean.
#' The conditional variance of the \code{n}th shock at time \code{t} is given by:
#' \deqn{Var_{t-1}[u_{n.t}] = s^2_{n.s_t}}
#' where \eqn{s_t} is a Markov process driving the time-variability of 
#' the regime-specific conditional variances of structural shocks \eqn{s^2_{n.s_t}}. 
#' In this model, the variances of each of the structural shocks sum to \code{M}.
#' 
#' The Markov process \eqn{s_t} is either:
#' \itemize{
#'   \item stationary, irreducible, and aperiodic which requires all regimes to have 
#'   a positive number occurrences over the sample period, or
#'   \item sparse with potentially many regimes with zero occurrences over the sample period
#'   and in which the number of regimes is estimated.
#' }
#' These model selection also with this respect is made using function \code{\link{specify_bsvar_msh}}.
#' 
#' @param S a positive integer, the number of posterior draws to be generated
#' @param specification an object of class BSVAR-MSH generated using the \code{specify_bsvar_msh$new()} function.
#' @param thin a positive integer, specifying the frequency of MCMC output thinning
#' @param show_progress a logical value, if \code{TRUE} the estimation progress bar is visible
#' 
#' @return An object of class PosteriorBSVAR-MSH containing the Bayesian estimation output and containing two elements:
#' 
#'  \code{posterior} a list with a collection of \code{S} draws from the posterior distribution generated via Gibbs sampler containing:
#'  \describe{
#'  \item{A}{an \code{NxKxS} array with the posterior draws for matrix \eqn{A}}
#'  \item{B}{an \code{NxNxS} array with the posterior draws for matrix \eqn{B}}
#'  \item{hyper}{a \code{5xS} matrix with the posterior draws for the hyper-parameters of the hierarchical prior distribution}
#'  \item{sigma2}{an \code{NxMxS} array with the posterior draws for the structural shocks conditional variances}
#'  \item{PR_TR}{an \code{MxMxS} array with the posterior draws for the transition matrix.}
#'  \item{xi}{an \code{MxTxS} array with the posterior draws for the regime allocation matrix.}
#'  \item{pi_0}{an \code{MxS} matrix with the posterior draws for the initial state probabilities}
#' }
#' 
#' \code{last_draw} an object of class BSVAR-MSH with the last draw of the current MCMC run as the starting value to be passed to the continuation of the MCMC estimation using \code{bsvar_msh()}. 
#' 
#' @seealso \code{\link{specify_bsvar_msh}}, \code{\link{specify_posterior_bsvar_msh}}, \code{\link{normalise_posterior}}
#'
#' @author Tomasz Woźniak \email{wozniak.tom@pm.me}
#' 
#' @references The model, prior distributions, and estimation algorithms were proposed by
#' 
#' Woźniak, T., and Droumaguet, M., (2022) Bayesian Assessment of Identifying Restrictions for Heteroskedastic Structural VARs
#' 
#' Some more analysis on heteroskedastic SVAR models was proposed by:
#' 
#' Lütkepohl, H., and Woźniak, T., (2020) Bayesian Inference for Structural Vector Autoregressions Identified by Markov-Switching Heteroskedasticity. \emph{Journal of Economic Dynamics and Control} \bold{113}, 103862, \doi{https://doi.org/10.1016/j.jedc.2020.103862}.
#' 
#' Sampling from the generalised-normal full conditional posterior distribution of matrix \eqn{B} is implemented using the Gibbs sampler by:
#' 
#' Waggoner, D.F., and Zha, T., (2003) A Gibbs sampler for structural vector autoregressions. \emph{Journal of Economic Dynamics and Control}, \bold{28}, 349--366, \doi{https://doi.org/10.1016/S0165-1889(02)00168-9}.
#'
#' Sampling from the multivariate normal full conditional posterior distribution of each of the \eqn{A} matrix row is implemented using the sampler by:
#' 
#' Chan, J.C.C., Koop, G, and Yu, X. (2021) Large Order-Invariant Bayesian VARs with Stochastic Volatility.
#' 
#' The estimation of the Markov-switching heteroskedasticity closely follows procedures described by:
#' 
#' Song, Y., and Woźniak, T., (2021) Markov Switching. \emph{Oxford Research Encyclopedia of Economics and Finance}, Oxford University Press, \doi{https://doi.org/10.1093/acrefore/9780190625979.013.174}.
#' 
#' and
#' 
#' Frühwirth-Schnatter, S., (2006) Finite Mixture and Markov Switching Models. Springer Series in Statistics. New York: Springer, \doi{https://doi.org/10.1007/978-0-387-35768-3}.
#' 
#' The sparse model is inspired by:
#' 
#' Malsiner-Walli, G., Frühwirth-Schnatter, S., and Grün, B. (2016) Model-based clustering based on sparse finite Gaussian mixtures. \emph{Statistics and Computing}, \bold{26}(1–2), 303--324, \doi{https://doi.org/10.1007/s11222-014-9500-2}.
#' 
#' The forward-filtering backward-sampling is implemented following the proposal by:
#' 
#' Chib, S. (1996) Calculating posterior distributions and modal estimates in Markov mixture models. \emph{Journal of Econometrics}, \bold{75}(1), 79–97, \doi{https://doi.org/10.1016/0304-4076(95)01770-4}.
#' 
#' @examples
#' # upload data
#' data(us_fiscal_lsuw)
#' 
#' # specify the model and set seed
#' specification  = specify_bsvar_msh$new(us_fiscal_lsuw, p = 4, M = 2)
#' set.seed(123)
#' 
#' # run the burn-in
#' burn_in        = estimate_bsvar_msh(50, specification)
#' 
#' # estimate the model
#' posterior      = estimate_bsvar_msh(100, burn_in$get_last_draw())
#' 
#' @export
estimate_bsvar_msh <- function(S, specification, thin = 10, show_progress = TRUE) {
  
  stopifnot("Argument S must be a positive integer number." = S > 1 & S %% 1 == 0)
  stopifnot("Argument specification must be of class BSVAR-MSH generated using the specify_bsvar_msh$new() function." = any(class(specification) == "BSVAR-MSH"))
  stopifnot("Argument thin must be a positive integer number." = thin > 0 & thin %% 1 == 0)
  
  prior               = specification$prior$get_prior()
  starting_values     = specification$starting_values$get_starting_values()
  VB                  = specification$identification$get_identification()
  data_matrices       = specification$data_matrices$get_data_matrices()
  finiteM             = specification$finiteM
  if (finiteM) {
    model             = "stationaryMSH"
  } else {
    model             = "sparseMSH"
  }
  
  qqq                 = .Call(`_bsvars_bsvar_msh_cpp`, S, data_matrices$Y, data_matrices$X, prior, VB, starting_values, thin, finiteM, TRUE, model, show_progress)
  
  specification$starting_values$set_starting_values(qqq$last_draw)
  output              = specify_posterior_bsvar_msh$new(specification, qqq$posterior)
  
  return(output)
}
