#' Estimate Squared Partial Correlation Coefficients
#' and Generate the Corresponding Sampling Distribution
#' Using Nonparametric Bootstrapping
#'
#' @author Ivan Jacob Agaloos Pesigan
#'
#' @details The vector of squared partial correlation coefficients
#'   (\eqn{r^{2}_{p}})
#'   is estimated from bootstrap samples.
#'   Confidence intervals are generated by obtaining
#'   percentiles corresponding to \eqn{100(1 - \alpha)\%}
#'   from the generated sampling
#'   distribution of \eqn{r^{2}_{p}},
#'   where \eqn{\alpha} is the significance level.
#'
#' @return Returns an object
#'   of class `betanb` which is a list with the following elements:
#'   \describe{
#'     \item{call}{Function call.}
#'     \item{args}{Function arguments.}
#'     \item{thetahatstar}{Sampling distribution of
#'       \eqn{r^{2}_{p}}.}
#'     \item{vcov}{Sampling variance-covariance matrix of
#'       \eqn{r^{2}_{p}}.}
#'     \item{est}{Vector of estimated
#'       \eqn{r^{2}_{p}}.}
#'     \item{fun}{Function used ("PCorNB").}
#'   }
#'
#' @inheritParams BetaNB
#'
#' @examples
#' # Data ---------------------------------------------------------------------
#' data("nas1982", package = "betaNB")
#'
#' # Fit Model in lm ----------------------------------------------------------
#' object <- lm(QUALITY ~ NARTIC + PCTGRT + PCTSUPP, data = nas1982)
#'
#' # NB -----------------------------------------------------------------------
#' nb <- NB(
#'   object,
#'   R = 100, # use a large value e.g., 5000L for actual research
#'   seed = 0508
#' )
#'
#' # PCorNB -------------------------------------------------------------------
#' out <- PCorNB(nb, alpha = 0.05)
#'
#' ## Methods -----------------------------------------------------------------
#' print(out)
#' summary(out)
#' coef(out)
#' vcov(out)
#' confint(out, level = 0.95)
#'
#' @family Beta Nonparametric Bootstrap Functions
#' @keywords betaNB pcor
#' @export
PCorNB <- function(object,
                   alpha = c(0.05, 0.01, 0.001)) {
  stopifnot(
    inherits(
      x = object,
      what = "nb"
    )
  )
  if (object$lm_process$p < 2) {
    stop("Two or more regressors is required.")
  }
  fun <- "PCorNB"
  est <- .PCorSq(
    srsq = .SPCor(
      betastar = object$lm_process$betastar,
      sigmacapx = object$lm_process$sigmacapx
    )^2,
    rsq = object$lm_process$rsq[1]
  )
  names(est) <- object$lm_process$xnames
  foo <- function(x) {
    sr <- .SPCor(
      betastar = .BetaStarofSigma(
        sigmacap = x,
        q = 1 / sqrt(diag(x)),
        k = object$lm_process$k
      ),
      sigmacapx = x[
        2:object$lm_process$k,
        2:object$lm_process$k,
        drop = FALSE
      ]
    )
    rsq <- .RSqofSigma(
      sigmacap = x,
      k = object$lm_process$k
    )
    return(
      .PCorSq(
        srsq = sr^2,
        rsq = rsq
      )
    )
  }
  thetahatstar <- lapply(
    X = object$thetahatstar,
    FUN = foo
  )
  vcov <- stats::var(
    do.call(
      what = "rbind",
      args = thetahatstar
    )
  )
  colnames(vcov) <- rownames(vcov) <- names(est)
  out <- list(
    call = match.call(),
    args = list(
      object = object,
      alpha = alpha
    ),
    thetahatstar = thetahatstar,
    jackknife = lapply(
      X = object$jackknife,
      FUN = foo
    ),
    vcov = vcov,
    est = est,
    fun = fun
  )
  class(out) <- c(
    "betanb",
    class(out)
  )
  return(
    out
  )
}
