% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sparsify.R
\name{sparsify}
\alias{sparsify}
\title{Extract the backbone from a network using a sparsification model}
\usage{
sparsify(
  U,
  s,
  escore,
  normalize,
  filter,
  symmetrize = TRUE,
  umst = FALSE,
  class = "original",
  narrative = FALSE
)
}
\arguments{
\item{U}{An unweighted unipartite graph, as: (1) an adjacency matrix in the form of a matrix or sparse \code{\link{Matrix}}; (2) an edgelist in the form of a two-column dataframe; (3) an \code{\link{igraph}} object.}

\item{s}{numeric: Sparsification parameter}

\item{escore}{string: Method for scoring edges' importance}

\item{normalize}{string: Method for normalizing edge scores}

\item{filter}{string: Type of filter to apply}

\item{symmetrize}{boolean: TRUE if the result should be symmetrized}

\item{umst}{boolean: TRUE if the backbone should include the union of minimum spanning trees, to ensure connectivity}

\item{class}{string: the class of the returned backbone graph, one of c("original", "matrix", "Matrix", "igraph", "edgelist").
If "original", the backbone graph returned is of the same class as \code{U}.}

\item{narrative}{boolean: TRUE if suggested text & citations should be displayed.}
}
\value{
An unweighted, undirected, unipartite graph of class \code{class}.
}
\description{
A generic function to extract the backbone of an undirected, unipartite
network using a sparsification model described by a combination of an edge scoring metric, a
edge score normalization, and an edge score filter.
}
\details{
The \code{escore} parameter determines how an unweighted edge's importance is calculated.
Unless noted below, scores are symmetric and larger values represent more important edges.
\itemize{
\item \code{random}: a random number drawn from a uniform distribution
\item \code{betweenness}: edge betweenness
\item \code{triangles}: number of triangles that include the edge
\item \code{jaccard}: jaccard similarity coefficient of the neighborhoods of an edge's endpoints, or alternatively, triangles normalized by the size of the union of the endpoints neighborhoods
\item \code{dice}: dice similarity coefficient of the neighborhoods of an edge's endpoints
\item \code{quadrangles}: number of quadrangles that include the edge
\item \verb{quadrilateral embeddedness}: geometric mean normalization of quadrangles
\item \code{degree}: degree of neighbor to which an edge is adjacent (asymmetric)
\item \code{meetmin}: triangles normalized by the smaller of the endpoints' neighborhoods' sizes
\item \code{geometric}: triangles normalized by the product of the endpoints' neighborhoods' sizes
\item \code{hypergeometric}: probability of the edge being included at least as many triangles if edges were random, given the size of the endpoints' neighborhoods (smaller is more important)
}

The \code{normalize} parameter determines whether edge scores are normalized.
\itemize{
\item \code{none}: no normalization is performed
\item \code{rank}: scores are normalized by neighborhood rank, such that the strongest edge in a node's neighborhood is ranked 1 (asymmetric)
\item \code{embeddedness}: scores are normalized using the maximum Jaccard coefficient of the top k-ranked neighbors of each endpoint, for all k
}

The \code{filter} parameter determines how edges are filtered based on their (normalized) edge scores.
\itemize{
\item \code{threshold}: Edges with scores >= \code{s} are retained in the backbone
\item \code{proportion}: Specifies the approximate proportion of edges to retain in the backbone
\item \code{degree}: Retains each node's d^\code{s} most important edges, where d is the node's degree (requires that \code{normalize = "rank"})
\item \code{disparity}: Applies the disparity filter using \code{\link[=disparity]{disparity()}}
}

Using \code{escore == "degree"} or \code{normalize == "rank"} can yield an assymmetric network. When \code{symmetrize == TRUE} (default),
after applying a filter, the network is symmetrized by such that i-j if i->j or i<-j.

Specific combinations of \code{escore}, \code{normalize}, \code{filter}, and \code{umst} correspond to specific sparsification models in the literature, and are available via the following wrapper functions:
\code{\link[=sparsify.with.skeleton]{sparsify.with.skeleton()}}, \code{\link[=sparsify.with.gspar]{sparsify.with.gspar()}}, \code{\link[=sparsify.with.lspar]{sparsify.with.lspar()}}, \code{\link[=sparsify.with.simmelian]{sparsify.with.simmelian()}}, \code{\link[=sparsify.with.jaccard]{sparsify.with.jaccard()}}, \code{\link[=sparsify.with.meetmin]{sparsify.with.meetmin()}}, \code{\link[=sparsify.with.geometric]{sparsify.with.geometric()}}, \code{\link[=sparsify.with.hypergeometric]{sparsify.with.hypergeometric()}}, \code{\link[=sparsify.with.localdegree]{sparsify.with.localdegree()}}, \code{\link[=sparsify.with.quadrilateral]{sparsify.with.quadrilateral()}}.
See the documentation for these wrapper functions for more details and the associated citation.
}
\examples{
U <- igraph::sample_sbm(60, matrix(c(.75,.25,.25,.25,.75,.25,.25,.25,.75),3,3), c(20,20,20))
plot(U) #A hairball
sparse <- sparsify(U, s = 0.6, escore = "jaccard", normalize = "rank",
filter = "degree", narrative = TRUE)
plot(sparse) #Clearly visible communities
}
\references{
{Neal, Z. P. (2022). backbone: An R Package to Extract Network Backbones. \emph{PLOS ONE, 17}, e0269137. \doi{10.1371/journal.pone.0269137}}
}
