#' @title Estimate tree height by DBH according to BWI3
#' @description Function calculates expected tree height given diameter in
#' breast height and species code
#' @param d13 numeric vector of diameter in breast height [cm]
#' @param sp TapeS species code, see also \code{\link{tprSpeciesCode}}
#' @param qtl desired quantile of height, either NULL (LS-regression) or one of
#' 17, 50, 83 (quantile regression)
#' @details Function evaluates the expected value of a Pettersen-Height Function
#' based on diameter in breast height and tree species code.
#' The Pettersen-Function (\eqn{h = 1.3 + (a + \frac{b}{dbh})^{-3}}) was fitted
#' on NFI 3 (BWI 3) data, using the main stand only.
#'
#' \code{d13} and \code{sp} should be of equal length or one of it can
#' be > 1 if the other is of length 1. Then, the shorter object will be extended
#' to match the length of the longer object. See examples.
#'
#' The quantile option return tree height at quantiles 17, 50 or 83. If
#' \code{qtl} is NULL, the result of a nonlinear least-squares regression is
#' provided.
#' @return a vector of same length as \code{d13} or \code{sp}, with tree
#' height in [m].
#'
#' @export
#'
#' @examples
#' sp <- 1
#' d13 <- 30
#' estHeight(d13, sp)
#'
#' sp <- 1
#' d13 <- seq(15, 50, 5)
#' estHeight(d13, sp)
#'
#' sp <- 1:36
#' d13 <- 30
#' estHeight(d13, sp)


estHeight <- function(d13, sp, qtl=NULL){

  ## check input
  if(!is.numeric(d13)) stop("'d13' must be numeric")
  if(any(d13 <= 0)) stop("'d13' must be > 0!")
  if(!is.numeric(sp)) stop("'sp' must be numeric")
  ## sp=37 refers to beech taper model based on pre-smoothed BDAT data
  sp <- ifelse(sp == 37, 15, sp)
  if(any(sp < 0) | any(sp > 36)) stop("'sp' must be > 0 and <= 36!")
  if(length(d13) != length(sp)){
    if(length(d13) > 1 & length(sp) == 1){
      sp <- rep(sp, length(d13))
    } else if(length(sp) > 1 & length(d13) == 1){
      d13 <- rep(d13, length(sp))
    } else {
      stop("length of 'd13' and 'sp' do not match!")
    }
  }
  df <- data.frame(no=1:length(d13), BaTPR=sp, d13=d13)

  ## get tree species coefficient table for mapping
  HtBaCoef <- HtCoef(sp, qtl)

  ## mapping of species code
  sp <- merge(df, HtBaCoef, by="BaTPR")
  sp <- sp[order(sp$no), c("d13", "a", "b")]

  #### calculate height ####
  ## get height
  ht <- sapply(1:nrow(sp), function(x){
    1.3 + 1 / (sp[x, "a"] + sp[x, "b"] / sp[x, "d13"])^3
  })
  return(ht)
}


#' @title returns coefficients for Pettersen-Height model
#' @description Function to provide model coefficients for Pettersen-height model
#' @param sp BDAT species code, could be NULL then all coefficients are returned
#' @param qtl quantile, either NULL or 17, 50, 83
#'
#' @return a data.frame with species code and coefficients
#'

HtCoef <- function(sp=NULL, qtl=NULL){

  if(is.null(qtl)){
    HtBaCoef <-
      structure(list(BaTPR = c(1L, 2L, 12L, 13L, 14L, 7L, 5L, 6L, 3L, 4L, 8L,
                                11L, 9L, 10L, 15L, 18L, 17L, 21L, 24L, 25L, 23L,
                                22L, 16L, 30L, 27L, 33L, 35L, 31L, 26L, 28L, 20L,
                                19L, 32L, 29L, 34L, 36L),
                     a = c(0.269544528674586, 0.269544528674586, 0.269544528674586,
                           0.269544528674586, 0.269544528674586, 0.293768226515212,
                           0.293768226515212, 0.293768226515212, 0.267076614067478,
                           0.261771510946731, 0.261771510946731, 0.277164782836491,
                           0.277164782836491, 0.277164782836491, 0.288382861810922,
                           0.288382861810922, 0.30166923259692, 0.290749061496993,
                           0.303188182798491, 0.303188182798491, 0.303188182798491,
                           0.303188182798491, 0.311972499288631, 0.311972499288631,
                           0.311972499288631, 0.311972499288631, 0.311972499288631,
                           0.311972499288631, 0.310190211381573, 0.309081776420583,
                           0.274323934345797, 0.274323934345797, 0.320600760130007,
                           0.320600760130007, 0.320600760130007, 0.320600760130007),
                     b = c(2.59622737786763, 2.59622737786763, 2.59622737786763,
                           2.59622737786763, 2.59622737786763, 1.97796413027279,
                           1.97796413027279, 1.97796413027279, 2.94046781008349,
                           2.82091527035482, 2.82091527035482, 2.27908890644349,
                           2.27908890644349, 2.27908890644349, 1.87086673849477,
                           1.87086673849477, 1.94348602651865, 1.69793006634948,
                           1.66395567185651, 1.66395567185651, 1.66395567185651,
                           1.66395567185651, 1.66666079522631, 1.66666079522631,
                           1.66666079522631, 1.66666079522631, 1.66666079522631,
                           1.66666079522631, 1.5322324282209, 1.69667198022082,
                           2.56810385879614, 2.56810385879614, 1.71159975813774,
                           1.71159975813774, 1.71159975813774, 1.71159975813774)),
                .Names = c("BaTPR", "a", "b"),
                row.names = c(NA, -36L),
                class = "data.frame")
  } else {

    HtBaCoef <- structure(list(
      BaTPR = rep(c(1L, 2L, 12L, 13L, 14L, 7L, 5L, 6L, 3L, 4L, 8L, 11L, 9L, 10L,
                     15L, 18L, 17L, 21L, 24L, 25L, 23L, 22L, 16L, 30L, 27L, 33L,
                     35L, 31L, 26L, 28L, 20L, 19L, 32L, 29L, 34L, 36L), each=3),
      Qtl = rep(c(17L, 50L, 83L), times=36),
      a = c(0.275065878755444, 0.268992192358335, 0.265006497589843, 0.275065878755444,
            0.268992192358335, 0.265006497589843, 0.275065878755444, 0.268992192358335,
            0.265006497589843, 0.275065878755444, 0.268992192358335, 0.265006497589843,
            0.275065878755444, 0.268992192358335, 0.265006497589843, 0.2939648510791,
            0.286266594989632, 0.306567580375132, 0.2939648510791, 0.286266594989632,
            0.306567580375132, 0.2939648510791, 0.286266594989632, 0.306567580375132,
            0.267855900066342, 0.267731212129673, 0.270704689837259, 0.267229084449139,
            0.258180973524463, 0.264562353405327, 0.267229084449139, 0.258180973524463,
            0.264562353405327, 0.285932349031885, 0.278836038033509, 0.26981369697455,
            0.285932349031885, 0.278836038033509, 0.26981369697455, 0.285932349031885,
            0.278836038033509, 0.26981369697455, 0.298984532063312, 0.288380087113951,
            0.279759203709734, 0.298984532063312, 0.288380087113951, 0.279759203709734,
            0.314191423014691, 0.301887338754929, 0.288788862693314, 0.291498857706007,
            0.303786607421949, 0.279942331733814, 0.291750009441035, 0.303740040382842,
            0.32177720015297, 0.291750009441035, 0.303740040382842, 0.32177720015297,
            0.291750009441035, 0.303740040382842, 0.32177720015297, 0.291750009441035,
            0.303740040382842, 0.32177720015297, 0.331769731715448, 0.31380142971051,
            0.299418571307071, 0.331769731715448, 0.31380142971051, 0.299418571307071,
            0.331769731715448, 0.31380142971051, 0.299418571307071, 0.331769731715448,
            0.31380142971051, 0.299418571307071, 0.331769731715448, 0.31380142971051,
            0.299418571307071, 0.331769731715448, 0.31380142971051, 0.299418571307071,
            0.310005854506512, 0.321060292573014, 0.300452207089158, 0.29633832224768,
            0.321771490231362, 0.308101851299269, 0.266560699377319, 0.286100274195141,
            0.278775172924244, 0.266560699377319, 0.286100274195141, 0.278775172924244,
            0.338922811873618, 0.319285861595816, 0.30367712402211, 0.338922811873618,
            0.319285861595816, 0.30367712402211, 0.338922811873618, 0.319285861595816,
            0.30367712402211, 0.338922811873618, 0.319285861595816, 0.30367712402211),
      b = c(3.03767871767823, 2.59900831983987, 2.24339420522131, 3.03767871767823,
            2.59900831983987, 2.24339420522131, 3.03767871767823, 2.59900831983987,
            2.24339420522131, 3.03767871767823, 2.59900831983987, 2.24339420522131,
            3.03767871767823, 2.59900831983987, 2.24339420522131, 1.95901847518644,
            1.8283800460082, 2.219013842975, 1.95901847518644, 1.8283800460082,
            2.219013842975, 1.95901847518644, 1.8283800460082, 2.219013842975,
            2.89256999969389, 2.31011124592237, 3.48729609885628, 3.26454743469268,
            2.41509755423984, 2.73194435411684, 3.26454743469268, 2.41509755423984,
            2.73194435411684, 2.59426030567833, 2.22178002948376, 2.01582616006309,
            2.59426030567833, 2.22178002948376, 2.01582616006309, 2.59426030567833,
            2.22178002948376,2.01582616006309, 2.26053326138267, 1.88380561677978,
            1.60483725420631, 2.26053326138267, 1.88380561677978, 1.60483725420631,
            2.4241037185018, 1.94954020752315, 1.75028662009817, 1.67562844076289,
            1.94417658644194, 1.50485024478897, 1.45316492279923, 1.65572699332348,
            1.83928968258963, 1.45316492279923, 1.65572699332348, 1.83928968258963,
            1.45316492279923, 1.65572699332348, 1.83928968258963, 1.45316492279923,
            1.65572699332348, 1.83928968258963, 1.99366604631888, 1.64980127983582,
            1.44064354965406, 1.99366604631888, 1.64980127983582, 1.44064354965406,
            1.99366604631888, 1.64980127983582, 1.44064354965406, 1.99366604631888,
            1.64980127983582, 1.44064354965406, 1.99366604631888, 1.64980127983582,
            1.44064354965406, 1.99366604631888, 1.64980127983582, 1.44064354965406,
            1.5380782047119, 1.84070940149534, 1.33932077451656, 1.55333244253334,
            2.03553854223844, 1.71327783878797, 2.19430024669513, 2.91306907326488,
            2.42098412205762, 2.19430024669513, 2.91306907326488, 2.42098412205762,
            2.17685809431446, 1.82512782955597, 1.55025692907092, 2.17685809431446,
            1.82512782955597, 1.55025692907092, 2.17685809431446, 1.82512782955597,
            1.55025692907092, 2.17685809431446, 1.82512782955597, 1.55025692907092)),
      .Names = c("BaTPR","Qtl", "a", "b"),
      class = "data.frame",
      row.names = c(NA, -108L))
    HtBaCoef <- HtBaCoef[HtBaCoef$Qtl == qtl[1], ]

  }

  if(!is.null(sp)){
    ## select those species, which should be returned
    HtBaCoef <- HtBaCoef[which(HtBaCoef$BaTPR %in% sp), ]
  }

  return(HtBaCoef)

}
