% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/lines_to_lsn.R
\name{lines_to_lsn}
\alias{lines_to_lsn}
\title{Convert lines to a landscape network (LSN)}
\usage{
lines_to_lsn(
  streams,
  lsn_path,
  check_topology = TRUE,
  snap_tolerance = 0,
  topo_tolerance = 0,
  remove_ZM = FALSE,
  overwrite = FALSE,
  use_parallel = FALSE,
  no_cores = NULL,
  verbose = TRUE
)
}
\arguments{
\item{streams}{An \code{sf} object with LINESTRING geometry
representing streams.}

\item{lsn_path}{Pathname to a directory in character format specifying
where to store the outputs. The directory will be created if it
does not already exist.}

\item{check_topology}{Logical. If \code{TRUE}, the edges will be
checked for topological errors and potential node errors will be
flagged in an output file called
\code{node_errors.gpkg}. Defaults to \code{TRUE}.}

\item{snap_tolerance}{Distance in map units >= 0. Two nodes separated by
a distance less than or equal to \code{snap_tolerance} are
assumed to be connected. Defaults to 0.}

\item{topo_tolerance}{Distance in map units >= 0. Only used if
\code{check_topology = TRUE}. Two nodes on the network separated
by a distance less than or equal to \code{topo_tolerance} are
flagged as potential topological errors in the network. Defaults
to 0.}

\item{remove_ZM}{Logical. If \code{TRUE}, Z and M dimensions are
removed from streams, if they exist. Default is \code{FALSE}.}

\item{overwrite}{Logical. If \code{TRUE}, output files will
overwrite existing files with the same names. If \code{FALSE} and
files sharing the same names as the outputs exist in the
\code{lsn_path}, the function will exit early with an error.}

\item{use_parallel}{Logical. if \code{TRUE} parallel processing will be used. Default is \code{FALSE}.}

\item{no_cores}{Integer representing the number of cores used when \code{use_parallel = TRUE}.}

\item{verbose}{Logical. If \code{TRUE}, messages describing
function progress will be printed to the console. Default is
\code{TRUE}.}
}
\value{
An \code{sf} object representing edges in the LSN. The LSN, including edges.gpkg, nodes.gpkg, nodexy.csv, noderelationships.csv, and relationships.csv files, are saved locally to a directory defined by \code{lsn_path}. If \code{check_topology = TRUE} and topological errors are identified, then node_errors.gpkg is also saved to \code{lsn_path}.
}
\description{
Convert an \code{sf} object containing features with
LINESTRING geometry to a landscape network (LSN), which is a
topological data model of streams/rivers represented as a
directional graph embedded in 2-D geographic space. Relationship
tables are created and topological relationships are checked.
}
\details{
\code{lines_to_lsn} converts an \code{sf} object
representing streams to a landscape network
(LSN), which is a directional graph used to represent
the topological and geographic relationships between line
features, along with additional attribute information.
\code{streams} must have LINESTRING geometry and a projected coordinate system, rather
than geographic coordinate system (i.e. not Longitude and Latitude).

The LSN is saved to a local directory defined by \code{lsn_path} and has 5 components:
\itemize{
\item{ nodes.gpkg: A GeoPackage of features with POINT geometry representing topologic breaks in the stream network such as pseudonodes, confluences, stream sources, or stream outlets. A column containing a unique node identifier, pointid, is included. }
\item{ edges.gpkg: A GeoPackage of features with LINESTRING geometry representing flow paths (i.e. line features) from node to node. A new column named rid is added that contains a unique identifier for each line feature. }
\item{nodexy.csv: a table with three columns: the pointid (unique node identifier), and the x-, y-coordinates for each node. }
\item{noderelationships.csv: a table describing the relationship between the nodes and their associated edges, as well as directionality in the LSN. The table contains three columns: the rid (for each edge), fromnode (pointid for the upstream node of each edge), and tonode (pointid for the downstream node of each edge). Directionality is defined based on the digitized direction of the line features. }
\item{relationships.csv: a table representing the downstream flow path from edge to edge. The table contains two columns: fromedge (upstream edge) and toedge (downstream edge). }
}

Topological errors are common in spatial data and must be corrected
to ensure that the LSN accurately represents direction and
connectivity within the stream network. When \code{check_topology =
TRUE}, the edges are checked for topological errors. Two nodes on
the network separated by a distance less than or equal to
\code{topo_tolerance} are flagged as potential topological errors
in the network saved in an output file called
\code{node_errors.gpkg}, which is also saved to \code{lsn_path}. In
addition to the pointid, \code{node_errors.gpkg} contains a column
containing the node class (nodecat), which can take on values of
Pseudonode, Confluence, Source, or Outlet. A second column (error)
defines the potential error type and can take on values of Complex
Confluence, Converging Node, Dangling Node, Intersection Without
Node, Downstream Divergence and Unsnapped Node. The nodecat column
is also added to nodes.gpkg. A node_errors.gpkg file is not
produced if no obvious errors are identified. There is no guarantee
that all topological errors will be identified and included in
node_errors.gpkg. Therefore, potential errors \emph{and} node classes
found in node_errors.gpkg and nodes.gpkg must be checked in a GIS
and topological errors in \code{streams} corrected before
rebuilding the LSN using \code{lines_to_lsn()}. This process is
iterative and must continue until the LSN is free of topological
errors.

The arguments \code{snap_tolerance} and \code{topo_tolerance} are
both provided in map units and must be >= 0. It is generally a good
idea to set the \code{snap_tolerance} to a relatively small value
compared to the \code{topo_tolerance} argument. Nodes separated by
a Euclidean distance <= \code{snap_tolerance} are assumed to be
connected.  If this distance <= (snap_tolerance/10000), the nodes
are automatically snapped when \code{check_topology = TRUE}.  When
(snap_tolerance/10000) <= distance <= snap_tolerance, the nodes are
not snapped and are instead flagged as potential errors for the
user to check and correct. Similarly, when snap_tolerance < distance <=
topo_tolerance, nodes are flagged as potential errors. Note that \code{snap_tolerance} must always be < the length of the shortest line feature found in \code{streams}. Use the \code{\link[sf]{st_length}} to obtain and check the length of each line feature.
}
