% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simulate_scenario.R
\name{simulate_scenario}
\alias{simulate_scenario}
\title{Simulate a scenario of household lifetime finances}
\usage{
simulate_scenario(
  household,
  portfolio,
  scenario_id = "default",
  current_date = get_current_date(),
  monte_carlo_samples = NULL,
  seeds = NULL,
  use_cache = FALSE,
  debug = FALSE,
  ...
)
}
\arguments{
\item{household}{An R6 object of class \code{Household}.}

\item{portfolio}{A nested \code{tibble} of class \code{Portfolio}.}

\item{scenario_id}{A character. ID of the scenario.}

\item{current_date}{A character. Current date in the format \code{YYYY-MM-DD}.
By default, it is the output of \code{\link[=get_current_date]{get_current_date()}}.}

\item{monte_carlo_samples}{An integer. Number of Monte Carlo samples.
If \code{NULL} (default), no Monte Carlo samples are generated.}

\item{seeds}{An integer vector. Seeds for the random number generator
used to generate random portfolio returns for each Monte Carlo sample.
If \code{NULL} (default), random seed is generated automatically.}

\item{use_cache}{A logical. If \code{TRUE}, the function uses memoised functions
to speed up the simulation. The results are cached in the folder
set by \code{\link[=set_cache]{set_cache()}}.}

\item{debug}{A logical. If \code{TRUE}, additional information is printed
during the simulation.}

\item{...}{Additional arguments passed simulation functions.}
}
\value{
A \code{tibble} with nested columns including:
\itemize{
\item \code{scenario_id} - (character) ID of the scenario
\item \code{sample} - (integer) ID of the Monte Carlo sample
\item \code{index} - (integer) year index starting from 0
\item \code{years_left} - (integer) years left to the end of the household lifespan
\item \code{date} - (date) date after \code{index} years from the current date
\item \code{year} - (integer) calendar year
\item \code{survival_prob} - (double) survival probability of the household
\item \code{members} - (nested tibble) data of each member in the household
\item \code{income} - (nested tibble) income streams
\item \code{total_income} - (double) total income of the household from all income streams
\item \code{spending} - (nested tibble) non-discretionary spending streams
\item \code{nondiscretionary_spending} - (double) total non-discretionary spending of the household from all non-discretionary spending streams
\item \code{human_capital} - (double) human capital of the household
\item \code{liabilities} - (double) liabilities of the household
\item \code{portfolio} - (nested tibble) state of investment portfolio
\item \code{financial_wealth} - (double) financial wealth of the household at the beginning of the year
\item \code{net_worth} - (double) net worth of the household
\item \code{discretionary_spending} - (double) optimal discretionary spending of the household
\item \code{total_spending} - (double) total spending of the household (discretionary + non-discretionary)
\item \code{financial_wealth_end} - (double) financial wealth of the household at the end of the year
\item \code{risk_tolerance} - (double) risk tolerance of the household
\item \code{smooth_consumption_preference} - (double) smooth consumption preference of the household
\item \code{consumption_impatience_preference} - (double) consumption impatience preference of the household
\item \code{time_value_discount} - (double) time value discount based on consumption impatience of the household
\item \code{discretionary_spending_utility} - (double) discretionary spending utility of the household based on the smooth consumption preference
\item \code{discretionary_spending_utility_weighted} - (double) discretionary spending utility of the household weighted by survival probability and time value discount.
}
}
\description{
The function simulates a scenario of household lifetime finances
and returns a \code{tibble} with nested columns.
By default no Monte Carlo samples are generated, and only
single sample based on portfolio expected returns are returned with
column \code{sample}=\code{0}.
If the additional Monte Carlo samples are generated, they have
consecutive IDs starting from 1 in the \code{sample} column.
}
\examples{
\dontshow{if (interactive()) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
older_member <- HouseholdMember$new(
  name       = "older",  
  birth_date = "1980-02-15",
  mode       = 80,
  dispersion = 10
)  
household <- Household$new()
household$add_member(older_member)  

household$expected_income <- list(
  "income" = c(
    "members$older$age <= 65 ~ 7000 * 12"
  )
)
household$expected_spending <- list(
  "spending" = c(
    "TRUE ~ 5000 * 12"
  )
)

portfolio <- create_portfolio_template() 
portfolio$accounts$taxable <- c(10000, 30000)
portfolio <- 
  portfolio |> 
  calc_effective_tax_rate(
    tax_rate_ltcg = 0.20, 
    tax_rate_ordinary_income = 0.40
  )

scenario <- 
  simulate_scenario(
   household = household,
   portfolio = portfolio,
   current_date = "2020-07-15"
  )
names(scenario)
\dontshow{\}) # examplesIf}
}
