% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/MRGoverlap.R
\name{MRGoverlap}
\alias{MRGoverlap}
\title{Function that finds and merges overlapping grid cells in a multi-resolution grid
The need for this function comes from an error in the gridding process, 
and it can be seen as symptom solving rather than solving the issue. 
This function will either just show the problematic grid cells or remove the overlaps.}
\usage{
MRGoverlap(himg, vars, himg2, action = "sum")
}
\arguments{
\item{himg}{The grid resulting from a call to multiResGrid}

\item{vars}{Variable(s) of interest that should be aggregated (necessary when ifg is
used for individual farm specific anonymization rules)}

\item{himg2}{A grid resulting from a call to multiResGrid}

\item{action}{How to treat the values of overlapping grid cells. Possible values are:
\describe{
\item{none}{return an \code{sf} data.frame just with the overlapping grid cells}
  \item{sum}{sum the values of the overlapping grid cells - NAs are ignored unless both cells
  are NA or one is NA and one is 0}
  \item{sumna}{sum the values of the overlapping grid cells - sum is NA if any of them is NA}
  \item{avg}{avg of the grid cells - NAs are ignored unless both cells
  are NA or one is NA and one is 0}
  \item{avgna}{avg of the grid cells - avg is NA if any of them is NA}
  \item{replace}{replace the problematic grid cells with grid cells from \code{himg2}}
}}
}
\value{
The function will return different objects, depending on "action". The returned object will
for different values of "action" be:
\describe{
  \item{none}{An \code{\link[sf]{sf}} data.frame with the overlapping grid cells}
  \item{sum}{A multi-resolution grid containing the sum the values of the overlapping grid cells.
     NAs are ignored unless both cells are NA or one is NA and one is 0}
  \item{sumna}{A multi-resolution grid containing the sum the values of the overlapping grid cells.
       The sum will be NA if any of them is NA}
  \item{avg}{A multi-resolution grid containing the average of the grid cells.
     NAs are ignored unless both cells are NA or one is NA and one is 0}
  \item{avgna}{A multi-resolution grid containing the average of the grid cells.
       The average will be NA if any of them is NA}
  \item{replace}{A multi-resolution grid containing, where the problematic grid cells
       area replaced with grid cells from \code{himg2}}
}
}
\description{
Function that finds and merges overlapping grid cells in a multi-resolution grid
The need for this function comes from an error in the gridding process, 
and it can be seen as symptom solving rather than solving the issue. 
This function will either just show the problematic grid cells or remove the overlaps.
}
\details{
A multi-resoluion grids should not have overlapping grid cells, by definition.  
However, this could happen through stitching different grids together. 
Although this should rather have been taken care of during the gridding
process, this is not always possible to redo for an end-user. 

This function can first of all be used to identify and show the 
overlapping grid cells. It can also be used to create a valid multi-resolution grid for these cases,
as long as the grd cells all have the same base grid (i.e. no overlapping grid cells are 
partly overlapping, whether the grid cells have the same size or not). 
However, there will be some additional errors introduced in this process. 

Except for \code{action = "none"} and \code{action = "replace"}, the function will 
try to create a valid grid based on the values in the grid. 
This means that it has to find a sensible value for the merged grid cells.
Frequently one of the grid cells will have an NA value, meaning that it has been 
suppressed. This means that there are observations in the grid cell, but relatively
few. If the overlapping grid cell has a value, this is most likely larger, it is
non-confidential. The default action is therefore to sum values, but ignore NA-values
unless both are NA or one is NA and the other is 0. The last case means that
one of the grid cells have a confidential number of records, whereas the 
other one has zero records. The total is then a confidential number of records.

if \code{action = "restart"}, there must be a second grid which includes the updated values of 
the overlapping grid cells. This could typically happen if the data set is too large
to be processed as a single batch. There could then be overlapping grid cells on the border 
between different batches. Instead of reprocessing the entire grid, it is possible to
reprocess the border regions (in one more more batches, as long as they are not overlapping).
The grid cells from the border regions have to be passed as \code{himg2}.

The function uses \code{\link[sf]{st_join}} to check for overlaps. If the grid has
a very high number of grid cells (a few tens of thousands), this process can be 
rather slow. In that case, it might be better to check parts of the grid separately.
}
\examples{
\donttest{
library(sf)
library(giscoR)
library(dplyr)

# These are SYNTHETIC agricultural FSS data 
data(ifs_dk) # Census data

# Create spatial data
ifg = fssgeo(ifs_dk, locAdj = "LL")

nuts2 = gisco_get_nuts(nuts_level = 2)
nuts2 = nuts2 \%>\% filter(CNTR_CODE == "DK") \%>\% st_transform(crs = st_crs(ifg))

ifg$NUTS2 = st_join(ifg, nuts2, join = st_nearest_feature)$NUTS_ID
ress = c(1,5,10,20,40, 80, 160)*1000
# Create regular grid of the variables, for three regions
ifl = gridData(ifg[ifg$NUTS2 \%in\% c("DK03", "DK04", "DK05"),], vars = c("UAA"), res = ress)
ifl3 = gridData(ifg[ifg$NUTS2 == "DK03",], vars = c("UAA"), res = ress)
ifl4 = gridData(ifg[ifg$NUTS2 == "DK04",], vars = c("UAA"), res = ress)
ifl5 = gridData(ifg[ifg$NUTS2 == "DK05",], vars = c("UAA"), res = ress)

# Create the different multi-resolution grids for different nuts regions
himg3 = multiResGrid(ifl3, vars = "UAA", ifg = ifg[ifg$NUTS2 == "DK03",], suppresslim = 0.02)
himg4 = multiResGrid(ifl4, vars = "UAA", ifg = ifg[ifg$NUTS2 == "DK04",], suppresslim = 0.02)
himg5 = multiResGrid(ifl5, vars = "UAA", ifg = ifg[ifg$NUTS2 == "DK05",], suppresslim = 0.02)

# Bind them together and create new consecutive IDs for the grid cells
himg = rbind(himg3, himg4, himg5)
himg$ID = 1:dim(himg)[1]

# Find the overlapping grid cells, and show some examples.
himgd = MRGoverlap(himg, action = "none")
dim(himgd)
himgd[himgd$ID.y \%in\% 932:940,]

# Remove overlapping grid cells
himgnew = MRGoverlap(himg, action = "sum")

# Check that there are no more overlaping grid cells
himgd2 = MRGoverlap(himgnew, action = "none")
himgd2


# Create a new multi-resolution grid which has the correct grid cells
# at the border. In this example, the region of interest is so small that
# it is difficult to reprocess just the border grid cells, so 
# we make a new complete grid

himg1 =  multiResGrid(ifl, vars = "UAA", ifg = ifg[ifg$NUTS2 \%in\% c("DK03", "DK04", "DK05"),],
                      suppresslim = 0.02)
himgnew2 = MRGoverlap(himg, himg2 = himg1, action = "replace")
himgd12 = MRGoverlap(himgnew2, action = "none")
himgd12


}


}
