% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/6-UserBayesFunctions.R
\name{sensbayes}
\alias{sensbayes}
\title{Verifying Optimality of Bayesian D-optimal Designs}
\usage{
sensbayes(formula, predvars, parvars, family = gaussian(), x, w, lx, ux,
  fimfunc = NULL, prior = list(), sens.bayes.control = list(),
  crt.bayes.control = list(), plot_3d = c("lattice", "rgl"),
  plot_sens = TRUE, npar = NULL, calculate_criterion = TRUE,
  silent = FALSE)
}
\arguments{
\item{formula}{A nonlinear model \code{\link[stats]{formula}}.
A symbolic description of the model consists of predictors and the unknown model parameters.
Will be coerced to a \code{\link[stats]{formula}} if necessary.}

\item{predvars}{A vector of characters. Denotes the predictors in the \code{\link[stats]{formula}}.}

\item{parvars}{A vector of characters. Denotes the unknown parameters in the \code{\link[stats]{formula}}.}

\item{family}{A description of the response distribution and link function to be used in the model.
This can be a family function, a call to a family function or a character string naming the family.
 Every family function has a link argument allowing to specify the link function to be applied on the response variable.
  If not specified, default links are used. For details see \code{\link[stats]{family}}.
   By default, a linear gaussian model \code{gaussian()} is applied.}

\item{x}{Vector of design (support) points. See 'Details' of \code{\link{sensminimax}}.}

\item{w}{Vector of corresponding design weights for \code{x}.}

\item{lx}{Vector of lower bounds for the predictors. Should be in the same order as \code{predvars}.}

\item{ux}{Vector of upper bounds for the predictors. Should be in the same order as \code{predvars}.}

\item{fimfunc}{A function. Returns the FIM as a \code{matrix}. Required when \code{formula} is missing. See 'Details' of \code{\link{minimax}}.}

\item{prior}{An object of class \code{cprior}. User can also use one of the functions
 \code{\link{uniform}}, \code{\link{normal}},
\code{\link{skewnormal}} or \code{\link{student}}  to create the  prior. See 'Details' of \code{\link{bayes}}.}

\item{sens.bayes.control}{Control parameters to verify the general equivalence theorem. For details, see \code{\link{sens.bayes.control}}.}

\item{crt.bayes.control}{Control parameters to approximate the integral in  Bayesian criterion at a given design over the parameter space.
For details, see \code{\link{crt.bayes.control}}.}

\item{plot_3d}{Which package should be used to plot the sensitivity (derivative) function for two-dimensional design space. Defaults to \code{"lattice"}.}

\item{plot_sens}{Plot the sensitivity (derivative) function? Defaults to \code{TRUE}.}

\item{npar}{Number of model parameters.  Used when \code{fimfunc} is given instead of \code{formula} to specify the number of model parameters.
If not specified truly, the sensitivity (derivative) plot may be shifted below the y-axis. When \code{NULL}, it will be set to \code{length(parvars)} or
\code{prior$npar} when \code{missing(formula)}.}

\item{calculate_criterion}{Evaluate the D-criterion? See 'Details' of \code{\link{sensminimax}}.}

\item{silent}{Do not print anything? Defaults to \code{FALSE}.}
}
\description{
This function plot the sensitivity (derivative) function given an approximate (continuous) design and calculate the efficiency lower bound (ELB) for Bayesian D-optimal designs.
Let \eqn{\boldsymbol{x}}{x} belongs to \eqn{\chi} that denotes the design space.
Based on the general equivalence theorem, generally, a design \eqn{\xi^*}{\xi*} is optimal if and only if the value of its sensitivity (derivative) function
be non-positive for all \eqn{\boldsymbol{x}}{x} in \eqn{\chi} and it only reaches zero
when \eqn{\boldsymbol{x}}{x} belong to the support of \eqn{\xi^*}{\xi*} (be equal to one of the design point).
Therefore, the user can look at the sensitivity plot and the ELB and decide whether the
design is optimal or close enough to the true optimal design (ELB tells us that without knowing the latter).
}
\details{
Let \eqn{\Xi} be the space of all  approximate designs with
 \eqn{k} design points (support points) at \eqn{x_1, x_2, ...,  x_k}{x1, x2, ...,  xk} from  design space \eqn{\chi} with
 corresponding weights  \eqn{w_1, . . . ,w_k}{w1, . . . ,wk}.
 Let \eqn{M(\xi, \theta)} be the Fisher information
  matrix (FIM) of a \eqn{k-}point design \eqn{\xi}
  and  \eqn{\pi(\theta)} is a user-given  prior distribution for the vector of unknown parameters \eqn{\theta}.
A design \eqn{\xi^*}{\xi*} is Bayesian D-optimal among all designs on \eqn{\chi} if and only if  the following inequality holds for all \eqn{\boldsymbol{x} \in \chi}{x belong to \chi}
 \deqn{c(\boldsymbol{x}, \xi^*) =  \int_{\theta \in Theta}tr M^{-1}(\xi^*, \theta)I(\boldsymbol{x}, \theta)-p \pi(\theta) d\theta\leq 0,}{
 c(x, \xi*) =  integration over \Theta tr M^-1(\xi*, \theta)I(x, \theta)-p <= 0,}
 with equality at all support points of \eqn{\xi^*}{\xi*}.
 Here, \eqn{p} is the number of model parameters.
 \eqn{c(\boldsymbol{x},\xi^*)}{c(x, \xi*)} is
  called \strong{sensitivity} or \strong{derivative} function.

 \strong{Sometimes, the CPU time can be considerably reduced
by choosing less conservative values for the tuning parameters \code{tol} and \code{maxEval} in
the function \code{\link{sens.bayes.control}}.}
The user should find a trade-off between accuracy and speed for his/her problem.
 See 'Examples'.
}
\note{
Having accurate plots for the sensitivity (derivative) function
 and calculating ELB to a high precision is the primary goal here,
  although the process may take too long (even hours) due to
requesting very accurate integral approximations.
}
\examples{
##################################################################
# Checking the Bayesian D-optimality of a design for the 2Pl model
##################################################################
skew2 <- skewnormal(xi = c(0, 1), Omega = matrix(c(1, -0.17, -0.17, .5), nrow = 2),
                    alpha = c(-1, 0), lower =  c(-3, .1), upper = c(3, 2))
\dontrun{
sensbayes(formula = ~1/(1 + exp(-b *(x - a))),
          predvars = "x", parvars = c("a", "b"),
          family = binomial(),
          x= c(-2.50914, -1.16780, -0.36904, 1.29227),
          w =c(0.35767, 0.11032, 0.15621, 0.37580),
          lx = -3, ux = 3,
          prior = skew2)
# took 29 seconds on my system!
}

# It took very long.
# We re-adjust the tuning parameters in sens.bayes.control to be faster
# See how we drastically reduce the maxEval and increase the tolerance
\dontrun{
sensbayes(formula = ~1/(1 + exp(-b *(x - a))),
          predvars = "x", parvars = c("a", "b"),
          family = binomial(),
          x= c(-2.50914, -1.16780, -0.36904, 1.29227),
          w =c(0.35767, 0.11032, 0.15621, 0.37580),
          lx = -3, ux = 3,prior = skew2,
          sens.bayes.control = list(cubature = list(tol = 1e-4, maxEval = 300)))
# took 5 Seconds on my system!
}



# Compare it with the following:
sensbayes(formula = ~1/(1 + exp(-b *(x - a))),
          predvars = "x", parvars = c("a", "b"),
          family = binomial(),
          x= c(-2.50914, -1.16780, -0.36904, 1.29227),
          w =c(0.35767, 0.11032, 0.15621, 0.37580),
          lx = -3, ux = 3,prior = skew2,
          sens.bayes.control = list(cubature = list(tol = 1e-4, maxEval = 200)))
# Look at the plot!
# took 3 seconds on my system


########################################################################################
# Checking the Bayesian D-optimality of a design for the 4-parameter sigmoid emax model
########################################################################################
lb <- c(4, 11, 100, 5)
ub <- c(9, 17, 140, 10)
\dontrun{
sensbayes(formula = ~ theta1 + (theta2 - theta1)*(x^theta4)/(x^theta4 + theta3^theta4),
          predvars = c("x"), parvars = c("theta1", "theta2", "theta3", "theta4"),
          x = c(0.78990, 95.66297, 118.42964,147.55809, 500),
          w = c(0.23426, 0.17071, 0.17684, 0.1827, 0.23549),
          lx = .001, ux = 500,  prior = uniform(lb, ub))
# took 200 seconds on my system
}

# Re-adjust the tuning parameters to have it faster
\dontrun{
sensbayes(formula = ~ theta1 + (theta2 - theta1)*(x^theta4)/(x^theta4 + theta3^theta4),
          predvars = c("x"), parvars = c("theta1", "theta2", "theta3", "theta4"),
          x = c(0.78990, 95.66297, 118.42964,147.55809, 500),
          w = c(0.23426, 0.17071, 0.17684, 0.1827, 0.23549),
          lx = .001, ux = 500,  prior = uniform(lb, ub),
          sens.bayes.control = list(cubature = list(tol = 1e-3, maxEval = 300)))
# took 4 seconds on my system. See how much it makes difference
}
}
