#' @export
#' @title Overlapping Generations Pure Exchange Models with Bank
#' @aliases gemOLG_PureExchange_Bank
#' @description Some examples of overlapping generations pure exchange models with bank.
#' Under a pay-as-you-go system, banks may only redistribute payoffs among consumers in each period.
#' This is, in each period a bank can get a part of payoff of age1 and pay it to age2 immediately.
#' From the consumer's point of view, the bank converts the current period's payoff into the next period's payoff.
#' Each consumer only transacts with the bank, and she can assume that there are no other consumers.
#' @param ... arguments to be passed to the function sdm2.
#' @seealso {
#' \code{\link{gemOLG_PureExchange}}
#' }
#' @examples
#' \donttest{
#' #### an example with a two-period-lived consumer
#' dst.bank <- node_new(
#'   "payoff2",
#'   type = "Leontief", a = 1,
#'   "payoff1"
#' )
#'
#' dst.consumer <- node_new(
#'   "util",
#'   type = "CD", alpha = 1, beta = c(1 / 2, 1 / 2),
#'   "payoff1", "payoff2"
#' )
#'
#' ge <- sdm2(
#'   A = list(dst.bank, dst.consumer),
#'   B = matrix(c(
#'     0, 0,
#'     1, 0
#'   ), 2, 2, TRUE),
#'   S0Exg = matrix(c(
#'     NA, 100,
#'     NA, NA
#'   ), 2, 2, TRUE),
#'   names.commodity = c("payoff1", "payoff2"),
#'   names.agent = c("bank", "consumer"),
#'   numeraire = "payoff1"
#' )
#'
#' ge$p
#' ge$D
#' ge$S
#'
#' #### an example with a three-period-lived consumer.
#' dst.bank1 <- node_new(
#'   "payoff2",
#'   type = "Leontief", a = 1,
#'   "payoff1"
#' )
#'
#' dst.bank2 <- node_new(
#'   "payoff3",
#'   type = "Leontief", a = 1,
#'   "payoff2"
#' )
#'
#' dst.consumer <- node_new(
#'   "util",
#'   type = "CD", alpha = 1, beta = c(1 / 3, 1 / 3, 1 / 3),
#'   "payoff1", "payoff2", "payoff3"
#' )
#'
#' ge <- sdm2(
#'   A = list(dst.bank1, dst.bank2, dst.consumer),
#'   B = matrix(c(
#'     0, 0, 0,
#'     1, 0, 0,
#'     0, 1, 0
#'   ), 3, 3, TRUE),
#'   S0Exg = matrix(c(
#'     NA, NA, 50,
#'     NA, NA, 50,
#'     NA, NA, NA
#'   ), 3, 3, TRUE),
#'   names.commodity = c("payoff1", "payoff2", "payoff3"),
#'   names.agent = c("bank1", "bank2", "consumer"),
#'   numeraire = "payoff1"
#' )
#'
#' ge$p
#' ge$S
#' ge$D
#'
#' ## Assume that banks can earn interest through foreign investment.
#' dst.bank1$a <- 0.8
#' dst.bank2$a <- 0.8
#' ge <- sdm2(
#'   A = list(dst.bank1, dst.bank2, dst.consumer),
#'   B = matrix(c(
#'     0, 0, 0,
#'     1, 0, 0,
#'     0, 1, 0
#'   ), 3, 3, TRUE),
#'   S0Exg = matrix(c(
#'     NA, NA, 50,
#'     NA, NA, 50,
#'     NA, NA, NA
#'   ), 3, 3, TRUE),
#'   names.commodity = c("payoff1", "payoff2", "payoff3"),
#'   names.agent = c("bank1", "bank2", "consumer"),
#'   numeraire = "payoff1"
#' )
#'
#' ge$p
#' ge$S
#' ge$D
#' ge$DV
#' }
#'

gemOLG_PureExchange_Bank <- function(...) sdm2(...)
