#' Coerce a fitted object to have `"xhaz"` as first class
#'
#' Ensures S3 dispatch hits \code{predict.xhaz()} by putting \code{"xhaz"} first
#' in the class vector.
#'
#' @param x An object with class vector containing \code{"xhaz"} (and usually \code{"mexhaz"}).
#' @return The same object with reordered classes.
#' @examples
#'
#' \donttest{
#' library(survival)
#' data("breast")
#' # The life table to be used is survexp.us. Note that SEX is coded 2 instead of female in survexp.us.
#' breast$sexe <- "female"
#'
#' fit.haz <- exphaz(
#'                   formula = Surv(temps, statut) ~ 1,
#'                   data = breast, ratetable = survexp.us,
#'                   only_ehazard = FALSE,
#'                   rmap = list(age = 'age', sex = 'sexe', year = 'date'))
#'
#' breast$expected <- fit.haz$ehazard
#'
#' res <- mexhaz(formula = Surv(temps, statut) ~ agecr + armt,
#'     data = breast, degree = 3,
#'      knots=quantile(breast[breast$statut==1,]$temps, probs=c(1:2/3)),
#'      expected = "expected", base = "exp.bs", random = "hosp")
#' resLT <- as.xhaz(res)
#' }
#'
#' @export
as.xhaz <- function(x) {
  x$recurrent <- FALSE
  class(x) <- c("xhaz", setdiff(class(x), "xhaz"))
  x
}

