% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ZINB.R
\name{ZINB}
\alias{ZINB}
\title{Fit Zero-Inflated Negative Binomial Model with Arbitrary Covariates and Prediction}
\usage{
ZINB(
  cases,
  pop = NULL,
  covariates_count = NULL,
  covariates_zero = NULL,
  covariatespred_count = NULL,
  covariatespred_zero = NULL,
  poppred = NULL,
  casespred = NULL,
  beta_init = NULL,
  delta_init = NULL,
  r_init = NULL,
  beta_prior_mean = 0,
  beta_prior_sd = 10,
  delta_prior_mean = 0,
  delta_prior_sd = 10,
  r_prior_shape = 1,
  r_prior_rate = 1,
  n_iter = 1e+05,
  n_burnin = 10000,
  n_chains = 3,
  n_thin = 1,
  save_params = c("beta", "delta", "r")
)
}
\arguments{
\item{cases}{Vector of observed counts (length N)}

\item{pop}{Optional vector of population offsets (length N)}

\item{covariates_count}{Optional numeric matrix (N x P) of covariates for the count component.}

\item{covariates_zero}{Optional numeric matrix (N x Q) of covariates for the zero-inflation component.}

\item{covariatespred_count}{Optional numeric matrix (M x P) of new covariates for count prediction.}

\item{covariatespred_zero}{Optional numeric matrix (M x Q) of new covariates for zero-inflation prediction.}

\item{poppred}{Optional vector of population offsets (length M) for prediction.}

\item{casespred}{Optional vector of true counts (length M) for prediction performance.}

\item{beta_init}{Optional list of length n_chains for beta, count coefficients initial values.}

\item{delta_init}{Optional list of length n_chains for delta, zero-inflation coefficients.}

\item{r_init}{Optional numeric vector of length n_chains for dispersion parameter.}

\item{beta_prior_mean}{Mean for beta prior (default: 0)}

\item{beta_prior_sd}{SD   for beta prior (default: 10)}

\item{delta_prior_mean}{Mean for delta prior (default: 0)}

\item{delta_prior_sd}{SD   for delta prior (default: 10)}

\item{r_prior_shape}{Shape for r ~ dgamma (default: 1)}

\item{r_prior_rate}{Rate  for r ~ dgamma (default: 1)}

\item{n_iter}{Total MCMC iterations (default: 100000)}

\item{n_burnin}{Burn-in iterations (default: 10000)}

\item{n_chains}{Number of chains (default: 3)}

\item{n_thin}{Thinning interval (default: 1)}

\item{save_params}{Character vector of parameters to save (default c("beta","delta","r"))}
}
\value{
A list with MCMC summary, samples, DIC, and if prediction data provided:
pred_matrix, pred_mean, mae, rmse
}
\description{
Fits a zero-inflated negative binomial (ZINB) model using JAGS, with an optional
design matrix of covariates and full inprod for mean structure, and
can generate posterior predictive counts for new covariate data.
}
\examples{
# ---- tiny example for users & CRAN (< 5s) ----
set.seed(5)
n <- 100
base  <- rnbinom(n, size = 5, mu = 7)
zeros <- rbinom(n, 1, 0.25)       # add extra zeros
cases <- ifelse(zeros == 1, 0L, base)
\dontshow{
stopifnot(length(cases) == n, all(cases >= 0))
}

# ---- actually fit the model, but only when JAGS is available ----
\dontshow{if (nzchar(Sys.which("jags")) && requireNamespace("R2jags", quietly = TRUE)) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
fit <- ZINB(
  cases = cases,
  # optional priors if your API exposes them, e.g.:
  # beta_prior_mean = 0, beta_prior_sd = 5,
  # r_prior_shape = 2,  r_prior_rate  = 0.5,
  n_iter   = 400,            # keep fast
  n_burnin = 200,
  n_chains = 1,
  n_thin   = 2
)
print(fit)

\donttest{
# ---- longer user-facing demo (skipped on checks) ----
if (nzchar(Sys.which("jags")) && requireNamespace("R2jags", quietly = TRUE)) {
  x <- sin(2*pi*seq_len(n)/12)       # simple seasonal regressor
  fit2 <- ZINB(
    cases = cases,
    covariates_count   = cbind(x),
    covariates_zero = cbind(x),
    n_iter   = 1000,
    n_burnin = 100,
    n_chains = 2,
    n_thin   = 2
  )
  print(fit2)
  # if a plot method exists:  # plot(fit2)
}
}

\dontrun{
# ---- time-consuming / full demo ----
if (nzchar(Sys.which("jags")) && requireNamespace("R2jags", quietly = TRUE)) {
  fit_full <- ZINB(
    cases = cases,
    n_iter   = 100000,
    n_burnin = 10000,
    n_chains = 4,
    n_thin   = 5
  )
  print(fit_full)
}
}

if (interactive()) {
  # e.g., plot(fit)
}
\dontshow{\}) # examplesIf}
}
