% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/wlr.R
\name{wlr}
\alias{wlr}
\alias{wlr.default}
\alias{wlr.tte_data}
\alias{wlr.counting_process}
\title{Weighted logrank test}
\usage{
wlr(data, weight, return_variance = FALSE, ratio = NULL, formula = NULL)

\method{wlr}{default}(data, weight, return_variance = FALSE, ratio = NULL, formula = NULL)

\method{wlr}{tte_data}(data, weight, return_variance = FALSE, ratio = NULL, formula = NULL)

\method{wlr}{counting_process}(data, weight, return_variance = FALSE, ratio = NULL, formula = NULL)
}
\arguments{
\item{data}{Dataset (generated by \code{\link[=sim_pw_surv]{sim_pw_surv()}}) that has been cut by
\code{\link[=counting_process]{counting_process()}}, \code{\link[=cut_data_by_date]{cut_data_by_date()}}, or \code{\link[=cut_data_by_event]{cut_data_by_event()}}.}

\item{weight}{Weighting functions, such as \code{\link[=fh]{fh()}}, \code{\link[=mb]{mb()}}, and
\code{\link[=early_zero]{early_zero()}}.}

\item{return_variance}{A logical flag that, if \code{TRUE}, adds columns
estimated variance for weighted sum of observed minus expected;
see details; Default: \code{FALSE}.}

\item{ratio}{randomization ratio (experimental:control).
\itemize{
\item If the \code{data} is generated by simtrial, such as
\itemize{
\item \code{data = sim_pw_surv(...) |> cut_data_by_date(...)}
\item \code{data = sim_pw_surv(...) |> cut_data_by_event(...)}
\item \code{data = sim_pw_surv(...) |> cut_data_by_date(...) |> counting_process(...)}
\item \code{data = sim_pw_surv(...) |> cut_data_by_event(...) |> counting_process(...)}
there is no need to input the \code{ratio}, as simtrial gets the \code{ratio} via the
\code{block} arguments in \code{\link[=sim_pw_surv]{sim_pw_surv()}}.
}
\item If the \code{data} is a custom dataset (see Example 2) below,
\itemize{
\item Users are suggested to input the planned randomization ratio to \code{ratio};
\item If not, simtrial takes the empirical randomization ratio.
}
}}

\item{formula}{A formula to specify the columns that contain the
time-to-event, event, treatment, and stratum variables. Only used by the
default S3 method because the other classes aleady have the required column
names. For stratified designs, the formula should have the form \code{Surv(tte, event) ~ treatment + strata(stratum)}, where \code{tte}, \code{event}, \code{treatment},
and \code{stratum} are the column names from \code{data} with the time-to-event
measurement, event status, treatment group, and stratum, respectively. For
unstratified designs, the formula can omit the stratum column: \code{Surv(tte, event) ~ treatment}.}
}
\value{
A list containing the test method (\code{method}),
parameters of this test method (\code{parameter}),
point estimate of the treatment effect (\code{estimate}),
standardized error of the treatment effect (\code{se}),
Z-score (\code{z}), p-values (\code{p_value}).
}
\description{
Weighted logrank test
}
\details{
\itemize{
\item \eqn{z} - Standardized normal Fleming-Harrington weighted logrank test.
\item \eqn{i} - Stratum index.
\item \eqn{d_i} - Number of distinct times at which events occurred in
stratum \eqn{i}.
\item \eqn{t_{ij}} - Ordered times at which events in stratum
\eqn{i}, \eqn{j = 1, 2, \ldots, d_i} were observed;
for each observation, \eqn{t_{ij}} represents the time post study entry.
\item \eqn{O_{ij.}} - Total number of events in stratum \eqn{i} that occurred
at time \eqn{t_{ij}}.
\item \eqn{O_{ije}} - Total number of events in stratum \eqn{i} in the
experimental treatment group that occurred at time \eqn{t_{ij}}.
\item \eqn{N_{ij.}} - Total number of study subjects in stratum \eqn{i}
who were followed for at least duration.
\item \eqn{E_{ije}} - Expected observations in experimental treatment group
given random selection of \eqn{O_{ij.}} from those in
stratum \eqn{i} at risk at time \eqn{t_{ij}}.
\item \eqn{V_{ije}} - Hypergeometric variance for \eqn{E_{ije}} as
produced in \code{Var} from \code{\link[=counting_process]{counting_process()}}.
\item \eqn{N_{ije}} - Total number of study subjects in
stratum \eqn{i} in the experimental treatment group
who were followed for at least duration \eqn{t_{ij}}.
\item \eqn{E_{ije}} - Expected observations in experimental group in
stratum \eqn{i} at time \eqn{t_{ij}} conditioning on the overall number
of events and at risk populations at that time and sampling at risk
observations without replacement:
\deqn{E_{ije} = O_{ij.} N_{ije}/N_{ij.}}
\item \eqn{S_{ij}} - Kaplan-Meier estimate of survival in combined
treatment groups immediately prior to time \eqn{t_{ij}}.
\item \eqn{\rho, \gamma} - Real parameters for Fleming-Harrington test.
\item \eqn{X_i} - Numerator for signed logrank test in stratum \eqn{i}
\deqn{X_i = \sum_{j=1}^{d_{i}} S_{ij}^\rho(1-S_{ij}^\gamma)(O_{ije}-E_{ije})}
\item \eqn{V_{ij}} - Variance used in denominator for Fleming-Harrington
weighted logrank tests
\deqn{V_i = \sum_{j=1}^{d_{i}} (S_{ij}^\rho(1-S_{ij}^\gamma))^2V_{ij})}
The stratified Fleming-Harrington weighted logrank test is then computed as:
\deqn{z = \sum_i X_i/\sqrt{\sum_i V_i}.}
}
}
\examples{
# ---------------------- #
#      Example 1         #
#  Use dataset generated #
#     by simtrial        #
# ---------------------- #
x <- sim_pw_surv(n = 200) |> cut_data_by_event(100)

# Example 1A: WLR test with FH wights
x |> wlr(weight = fh(rho = 0, gamma = 0.5))
x |> wlr(weight = fh(rho = 0, gamma = 0.5), return_variance = TRUE)

# Example 1B: WLR test with MB wights
x |> wlr(weight = mb(delay = 4, w_max = 2))

# Example 1C: WLR test with early zero wights
x |> wlr(weight = early_zero(early_period = 4))

# Example 1D
# For increased computational speed when running many WLR tests, you can
# pre-compute the counting_process() step first, and then pass the result of
# counting_process() directly to wlr()
x <- x |> counting_process(arm = "experimental")
x |> wlr(weight = fh(rho = 0, gamma = 1))
x |> wlr(weight = mb(delay = 4, w_max = 2))
x |> wlr(weight = early_zero(early_period = 4))

# ---------------------- #
#      Example 2         #
#  Use cumsum dataset    #
# ---------------------- #
x <- data.frame(treatment = ifelse(ex1_delayed_effect$trt == 1, "experimental", "control"),
                stratum = rep("All", nrow(ex1_delayed_effect)),
                tte = ex1_delayed_effect$month,
                event = ex1_delayed_effect$evntd)

# Users can specify the randomization ratio to calculate the statistical information under H0
x |> wlr(weight = fh(rho = 0, gamma = 0.5), ratio = 2)

x |>
  counting_process(arm = "experimental") |>
  wlr(weight = fh(rho = 0, gamma = 0.5), ratio = 2)

# If users don't provide the randomization ratio, we will calculate the emperical ratio
x |> wlr(weight = fh(rho = 0, gamma = 0.5))

x |>
  counting_process(arm = "experimental") |>
  wlr(weight = fh(rho = 0, gamma = 0.5))

# ---------------------- #
#      Example 3         #
#  Use formula           #
# ---------------------- #
library("survival")

# Unstratified design
x <- sim_pw_surv(n = 200) |> cut_data_by_event(100) |> as.data.frame()
colnames(x) <- c("tte", "evnt", "strtm", "trtmnt")
wlr(x, weight = fh(0, 0.5), formula = Surv(tte, evnt) ~ trtmnt)

# Stratified design
x$strtm <- sample(c("s1", "s2"), size = nrow(x), replace = TRUE)
wlr(x, weight = fh(0, 0.5), formula = Surv(tte, evnt) ~ trtmnt + strata(strtm))
}
