/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5HFmodule.h" 
#define H5HF_TESTING    

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5HFpkg.h"     
#include "H5MMprivate.h" 

herr_t
H5HF_get_cparam_test(const H5HF_t *fh, H5HF_create_t *cparam)
{
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(fh);
    assert(cparam);

    
    if (fh->hdr->id_len == (unsigned)(1 + fh->hdr->heap_off_size + fh->hdr->heap_len_size))
        cparam->id_len = 0;
    else if (fh->hdr->id_len == (unsigned)(1 + fh->hdr->sizeof_size + fh->hdr->sizeof_addr))
        cparam->id_len = 1;
    else
        H5_CHECKED_ASSIGN(cparam->id_len, uint16_t, fh->hdr->id_len, unsigned);
    cparam->max_man_size = fh->hdr->max_man_size;
    H5MM_memcpy(&(cparam->managed), &(fh->hdr->man_dtable.cparam), sizeof(H5HF_dtable_cparam_t));
    H5O_msg_copy(H5O_PLINE_ID, &(fh->hdr->pline), &(cparam->pline));

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

int
H5HF_cmp_cparam_test(const H5HF_create_t *cparam1, const H5HF_create_t *cparam2)
{
    int ret_value = 0; 

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(cparam1);
    assert(cparam2);

    
    if (cparam1->managed.width < cparam2->managed.width)
        HGOTO_DONE(-1);
    else if (cparam1->managed.width > cparam2->managed.width)
        HGOTO_DONE(1);
    if (cparam1->managed.start_block_size < cparam2->managed.start_block_size)
        HGOTO_DONE(-1);
    else if (cparam1->managed.start_block_size > cparam2->managed.start_block_size)
        HGOTO_DONE(1);
    if (cparam1->managed.max_direct_size < cparam2->managed.max_direct_size)
        HGOTO_DONE(-1);
    else if (cparam1->managed.max_direct_size > cparam2->managed.max_direct_size)
        HGOTO_DONE(1);
    if (cparam1->managed.max_index < cparam2->managed.max_index)
        HGOTO_DONE(-1);
    else if (cparam1->managed.max_index > cparam2->managed.max_index)
        HGOTO_DONE(1);
    if (cparam1->managed.start_root_rows < cparam2->managed.start_root_rows)
        HGOTO_DONE(-1);
    else if (cparam1->managed.start_root_rows > cparam2->managed.start_root_rows)
        HGOTO_DONE(1);

    
    if (cparam1->max_man_size < cparam2->max_man_size)
        HGOTO_DONE(-1);
    else if (cparam1->max_man_size > cparam2->max_man_size)
        HGOTO_DONE(1);
    if (cparam1->id_len < cparam2->id_len)
        HGOTO_DONE(-1);
    else if (cparam1->id_len > cparam2->id_len)
        HGOTO_DONE(1);

    
    if (cparam1->pline.nused < cparam2->pline.nused)
        HGOTO_DONE(-1);
    else if (cparam1->pline.nused > cparam2->pline.nused)
        HGOTO_DONE(1);
    else {
        size_t u, v; 

        
        for (u = 0; u < cparam1->pline.nused; u++) {
            
            if (cparam1->pline.filter[u].id < cparam2->pline.filter[u].id)
                HGOTO_DONE(-1);
            else if (cparam1->pline.filter[u].id > cparam2->pline.filter[u].id)
                HGOTO_DONE(1);

            
            if (cparam1->pline.filter[u].flags < cparam2->pline.filter[u].flags)
                HGOTO_DONE(-1);
            else if (cparam1->pline.filter[u].flags > cparam2->pline.filter[u].flags)
                HGOTO_DONE(1);

#if 0
            
            if(!cparam1->pline.filter[u].name && cparam2->pline.filter[u].name)
                HGOTO_DONE(-1);
            else if(cparam1->pline.filter[u].name && !cparam2->pline.filter[u].name)
                HGOTO_DONE(1);
            else if(cparam1->pline.filter[u].name && cparam2->pline.filter[u].name) {
                if((ret_value = strcmp(cparam1->pline.filter[u].name, cparam2->pline.filter[u].name)))
                    HGOTO_DONE(ret_value);
            } 
#endif

            
            if (cparam1->pline.filter[u].cd_nelmts < cparam2->pline.filter[u].cd_nelmts)
                HGOTO_DONE(-1);
            else if (cparam1->pline.filter[u].cd_nelmts > cparam2->pline.filter[u].cd_nelmts)
                HGOTO_DONE(1);

            
            for (v = 0; v < cparam1->pline.filter[u].cd_nelmts; v++) {
                if (cparam1->pline.filter[u].cd_values[v] < cparam2->pline.filter[u].cd_values[v])
                    HGOTO_DONE(-1);
                else if (cparam1->pline.filter[u].cd_values[v] > cparam2->pline.filter[u].cd_values[v])
                    HGOTO_DONE(1);
            } 

        } 
    }     

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

unsigned
H5HF_get_max_root_rows(const H5HF_t *fh)
{
    unsigned ret_value = 0; 

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(fh);

    
    ret_value = fh->hdr->man_dtable.max_root_rows;

    FUNC_LEAVE_NOAPI(ret_value)
} 

unsigned
H5HF_get_dtable_width_test(const H5HF_t *fh)
{
    unsigned ret_value = 0; 

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(fh);

    
    ret_value = fh->hdr->man_dtable.cparam.width;

    FUNC_LEAVE_NOAPI(ret_value)
} 

unsigned
H5HF_get_dtable_max_drows_test(const H5HF_t *fh)
{
    unsigned ret_value = 0; 

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(fh);

    
    ret_value = fh->hdr->man_dtable.max_direct_rows;

    FUNC_LEAVE_NOAPI(ret_value)
} 

unsigned
H5HF_get_iblock_max_drows_test(const H5HF_t *fh, unsigned pos)
{
    unsigned ret_value = 0; 

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(fh);
    assert(pos);

    
    ret_value = pos + (fh->hdr->man_dtable.max_direct_bits - fh->hdr->man_dtable.first_row_bits) + 1;

    FUNC_LEAVE_NOAPI(ret_value)
} 

hsize_t
H5HF_get_dblock_size_test(const H5HF_t *fh, unsigned row)
{
    hsize_t ret_value = 0; 

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(fh);

    
    ret_value = fh->hdr->man_dtable.row_block_size[row];

    FUNC_LEAVE_NOAPI(ret_value)
} 

hsize_t
H5HF_get_dblock_free_test(const H5HF_t *fh, unsigned row)
{
    hsize_t ret_value = 0; 

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(fh);

    
    ret_value = fh->hdr->man_dtable.row_tot_dblock_free[row];

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF_get_id_off_test(const H5HF_t *fh, const void *_id, hsize_t *obj_off)
{
    const uint8_t *id = (const uint8_t *)_id; 

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(fh);
    assert(fh->hdr);
    assert(id);
    assert(obj_off);

    
    id++;
    UINT64DECODE_VAR(id, *obj_off, fh->hdr->heap_off_size);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5HF_get_id_type_test(const void *_id, unsigned char *obj_type)
{
    const uint8_t *id = (const uint8_t *)_id; 

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(id);
    assert(obj_type);

    
    *obj_type = (uint8_t)(*id & H5HF_ID_TYPE_MASK);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5HF_get_tiny_info_test(const H5HF_t *fh, size_t *max_len, bool *len_extended)
{
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(fh);
    assert(fh->hdr);
    assert(max_len);
    assert(len_extended);

    
    *max_len      = fh->hdr->tiny_max_len;
    *len_extended = fh->hdr->tiny_len_extended;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5HF_get_huge_info_test(const H5HF_t *fh, hsize_t *next_id, bool *ids_direct)
{
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    assert(fh);
    assert(fh->hdr);
    assert(ids_direct);

    
    if (next_id)
        *next_id = fh->hdr->huge_next_id;
    *ids_direct = fh->hdr->huge_ids_direct;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 
