/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Gmodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5Fprivate.h"  
#include "H5Gpkg.h"      
#include "H5HLprivate.h" 
#include "H5MMprivate.h" 

typedef struct {
    
    const char *name; 
    H5HL_t     *heap; 

    
    H5O_link_t *lnk; 
} H5G_stab_fnd_ud_t;

typedef struct H5G_bt_it_gnbi_t {
    
    H5G_bt_it_idx_common_t common; 
    H5HL_t                *heap;   

    
    char *name; 
} H5G_bt_it_gnbi_t;

#ifndef H5_NO_DEPRECATED_SYMBOLS

typedef struct H5G_bt_it_gtbi_t {
    
    H5G_bt_it_idx_common_t common; 
    H5F_t                 *f;      

    
    H5G_obj_t type; 
} H5G_bt_it_gtbi_t;
#endif 

typedef struct H5G_bt_it_lbi_t {
    
    H5G_bt_it_idx_common_t common; 
    H5HL_t                *heap;   

    
    H5O_link_t *lnk;   
    bool        found; 
} H5G_bt_it_lbi_t;

herr_t
H5G__stab_create_components(H5F_t *f, H5O_stab_t *stab, size_t size_hint)
{
    H5HL_t *heap = NULL;         
    size_t  name_offset;         
    herr_t  ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(stab);
    assert(size_hint > 0);

    
    if (H5B_create(f, H5B_SNODE, NULL, &(stab->btree_addr) ) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTINIT, FAIL, "can't create B-tree");

    
    if (H5HL_create(f, size_hint, &(stab->heap_addr) ) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTINIT, FAIL, "can't create heap");

    
    if (NULL == (heap = H5HL_protect(f, stab->heap_addr, H5AC__NO_FLAGS_SET)))
        HGOTO_ERROR(H5E_SYM, H5E_CANTPROTECT, FAIL, "unable to protect symbol table heap");

    
    if (H5HL_insert(f, heap, (size_t)1, "", &name_offset) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTINSERT, FAIL, "can't insert name into heap");

    
    assert(0 == name_offset);

done:
    
    if (heap && FAIL == H5HL_unprotect(heap))
        HDONE_ERROR(H5E_SYM, H5E_CANTUNPROTECT, FAIL, "unable to unprotect symbol table heap");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5G__stab_create(H5O_loc_t *grp_oloc, const H5O_ginfo_t *ginfo, H5O_stab_t *stab)
{
    size_t heap_hint;           
    size_t size_hint;           
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE_TAG(grp_oloc->addr)

    
    assert(grp_oloc);
    assert(stab);

    
    if (ginfo->lheap_size_hint == 0)
        heap_hint =
            8 + 
            (ginfo->est_num_entries *
             H5HL_ALIGN(ginfo->est_name_len +
                        1)) + 
            H5HL_SIZEOF_FREE(grp_oloc->file); 
    else
        heap_hint = ginfo->lheap_size_hint;
    size_hint = MAX(heap_hint, H5HL_SIZEOF_FREE(grp_oloc->file) + 2);

    
    if (H5G__stab_create_components(grp_oloc->file, stab, size_hint) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTINIT, FAIL, "can't create symbol table components");

    
    if (H5O_msg_create(grp_oloc, H5O_STAB_ID, 0, H5O_UPDATE_TIME, stab) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTINIT, FAIL, "can't create message");

done:
    FUNC_LEAVE_NOAPI_TAG(ret_value)
} 

herr_t
H5G__stab_insert_real(H5F_t *f, const H5O_stab_t *stab, H5O_link_t *obj_lnk, H5O_type_t obj_type,
                      const void *crt_info)
{
    H5HL_t      *heap = NULL;         
    H5G_bt_ins_t udata;               
    herr_t       ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(stab);
    assert(obj_lnk);

    
    if (NULL == (heap = H5HL_protect(f, stab->heap_addr, H5AC__NO_FLAGS_SET)))
        HGOTO_ERROR(H5E_SYM, H5E_CANTPROTECT, FAIL, "unable to protect symbol table heap");

    
    udata.common.name       = obj_lnk->name;
    udata.common.heap       = heap;
    udata.common.block_size = H5HL_heap_get_size(heap);
    udata.lnk               = obj_lnk;
    udata.obj_type          = obj_type;
    udata.crt_info          = crt_info;

    
    if (H5B_insert(f, H5B_SNODE, stab->btree_addr, &udata) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTINSERT, FAIL, "unable to insert entry");

done:
    
    if (heap && H5HL_unprotect(heap) < 0)
        HDONE_ERROR(H5E_SYM, H5E_CANTUNPROTECT, FAIL, "unable to unprotect symbol table heap");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5G__stab_insert(const H5O_loc_t *grp_oloc, H5O_link_t *obj_lnk, H5O_type_t obj_type, const void *crt_info)
{
    H5O_stab_t stab;                
    herr_t     ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(grp_oloc && grp_oloc->file);
    assert(obj_lnk);

    
    if (NULL == H5O_msg_read(grp_oloc, H5O_STAB_ID, &stab))
        HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "not a symbol table");

    if (H5G__stab_insert_real(grp_oloc->file, &stab, obj_lnk, obj_type, crt_info) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTINSERT, H5_ITER_ERROR, "unable to insert the link");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5G__stab_remove(const H5O_loc_t *loc, H5RS_str_t *grp_full_path_r, const char *name)
{
    H5HL_t     *heap = NULL;         
    H5O_stab_t  stab;                
    H5G_bt_rm_t udata;               
    herr_t      ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    assert(loc && loc->file);
    assert(name && *name);

    
    if (NULL == H5O_msg_read(loc, H5O_STAB_ID, &stab))
        HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "not a symbol table");

    
    if (NULL == (heap = H5HL_protect(loc->file, stab.heap_addr, H5AC__NO_FLAGS_SET)))
        HGOTO_ERROR(H5E_SYM, H5E_CANTPROTECT, FAIL, "unable to protect symbol table heap");

    
    udata.common.name       = name;
    udata.common.heap       = heap;
    udata.common.block_size = H5HL_heap_get_size(heap);
    udata.grp_full_path_r   = grp_full_path_r;

    
    if (H5B_remove(loc->file, H5B_SNODE, stab.btree_addr, &udata) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTREMOVE, FAIL, "unable to remove entry");

done:
    
    if (heap && H5HL_unprotect(heap) < 0)
        HDONE_ERROR(H5E_SYM, H5E_CANTUNPROTECT, FAIL, "unable to unprotect symbol table heap");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5G__stab_remove_by_idx(const H5O_loc_t *grp_oloc, H5RS_str_t *grp_full_path_r, H5_iter_order_t order,
                        hsize_t n)
{
    H5HL_t     *heap = NULL;          
    H5O_stab_t  stab;                 
    H5G_bt_rm_t udata;                
    H5O_link_t  obj_lnk;              
    bool        lnk_copied = false;   
    herr_t      ret_value  = SUCCEED; 

    FUNC_ENTER_PACKAGE

    assert(grp_oloc && grp_oloc->file);

    
    if (H5G__stab_lookup_by_idx(grp_oloc, order, n, &obj_lnk) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "can't get link information");
    lnk_copied = true;

    
    if (NULL == H5O_msg_read(grp_oloc, H5O_STAB_ID, &stab))
        HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "not a symbol table");

    
    if (NULL == (heap = H5HL_protect(grp_oloc->file, stab.heap_addr, H5AC__NO_FLAGS_SET)))
        HGOTO_ERROR(H5E_SYM, H5E_CANTPROTECT, FAIL, "unable to protect symbol table heap");

    
    udata.common.name       = obj_lnk.name;
    udata.common.heap       = heap;
    udata.common.block_size = H5HL_heap_get_size(heap);
    udata.grp_full_path_r   = grp_full_path_r;

    
    if (H5B_remove(grp_oloc->file, H5B_SNODE, stab.btree_addr, &udata) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTREMOVE, FAIL, "unable to remove entry");

done:
    
    if (heap && H5HL_unprotect(heap) < 0)
        HDONE_ERROR(H5E_SYM, H5E_CANTUNPROTECT, FAIL, "unable to unprotect symbol table heap");

    
    if (lnk_copied)
        H5O_msg_reset(H5O_LINK_ID, &obj_lnk);

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5G__stab_delete(H5F_t *f, const H5O_stab_t *stab)
{
    H5HL_t     *heap = NULL;         
    H5G_bt_rm_t udata;               
    herr_t      ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    assert(f);
    assert(stab);
    assert(H5_addr_defined(stab->btree_addr));
    assert(H5_addr_defined(stab->heap_addr));

    
    if (NULL == (heap = H5HL_protect(f, stab->heap_addr, H5AC__NO_FLAGS_SET)))
        HGOTO_ERROR(H5E_SYM, H5E_CANTPROTECT, FAIL, "unable to protect symbol table heap");

    
    udata.common.name = NULL;
    udata.common.heap = heap;

    
    if (H5B_delete(f, H5B_SNODE, stab->btree_addr, &udata) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTDELETE, FAIL, "unable to delete symbol table B-tree");

    
    if (H5HL_unprotect(heap) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTUNPROTECT, FAIL, "unable to unprotect symbol table heap");
    heap = NULL;

    
    if (H5HL_delete(f, stab->heap_addr) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTDELETE, FAIL, "unable to delete symbol table heap");

done:
    
    if (heap && H5HL_unprotect(heap) < 0)
        HDONE_ERROR(H5E_SYM, H5E_CANTUNPROTECT, FAIL, "unable to unprotect symbol table heap");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5G__stab_iterate(const H5O_loc_t *oloc, H5_iter_order_t order, hsize_t skip, hsize_t *last_lnk,
                  H5G_lib_iterate_t op, void *op_data)
{
    H5HL_t          *heap = NULL;           
    H5O_stab_t       stab;                  
    H5G_link_table_t ltable    = {0, NULL}; 
    herr_t           ret_value = FAIL;      

    FUNC_ENTER_PACKAGE

    
    assert(oloc);
    assert(op);

    
    if (NULL == H5O_msg_read(oloc, H5O_STAB_ID, &stab))
        HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "unable to determine local heap address");

    
    if (NULL == (heap = H5HL_protect(oloc->file, stab.heap_addr, H5AC__READ_ONLY_FLAG)))
        HGOTO_ERROR(H5E_SYM, H5E_CANTPROTECT, FAIL, "unable to protect symbol table heap");

    
    
    if (order != H5_ITER_DEC) {
        H5G_bt_it_it_t udata; 

        
        udata.heap      = heap;
        udata.skip      = skip;
        udata.final_ent = last_lnk;
        udata.op        = op;
        udata.op_data   = op_data;

        
        if ((ret_value = H5B_iterate(oloc->file, H5B_SNODE, stab.btree_addr, H5G__node_iterate, &udata)) < 0)
            HERROR(H5E_SYM, H5E_CANTNEXT, "iteration operator failed");

        
        
        if (skip > 0 && skip >= *last_lnk)
            HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid index specified");
    } 
    else {
        H5G_bt_it_bt_t udata; 

        
        udata.alloc_nlinks = 0;
        udata.heap         = heap;
        udata.ltable       = &ltable;

        
        if (H5B_iterate(oloc->file, H5B_SNODE, stab.btree_addr, H5G__node_build_table, &udata) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_BADITER, FAIL, "unable to build link table");

        
        if (skip > 0 && (size_t)skip >= ltable.nlinks)
            HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "index out of bound");

        
        if (H5G__link_sort_table(&ltable, H5_INDEX_NAME, order) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_CANTSORT, FAIL, "error sorting link messages");

        
        if ((ret_value = H5G__link_iterate_table(&ltable, skip, last_lnk, op, op_data)) < 0)
            HERROR(H5E_SYM, H5E_CANTNEXT, "iteration operator failed");
    } 

done:
    
    if (heap && H5HL_unprotect(heap) < 0)
        HDONE_ERROR(H5E_SYM, H5E_CANTUNPROTECT, FAIL, "unable to unprotect symbol table heap");
    if (ltable.lnks && H5G__link_release_table(&ltable) < 0)
        HDONE_ERROR(H5E_SYM, H5E_CANTRELEASE, FAIL, "unable to release link table");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5G__stab_count(const H5O_loc_t *oloc, hsize_t *num_objs)
{
    H5O_stab_t stab; 
    herr_t     ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE_TAG(oloc->addr)

    
    assert(oloc);
    assert(num_objs);

    
    *num_objs = 0;

    
    if (NULL == H5O_msg_read(oloc, H5O_STAB_ID, &stab))
        HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "unable to determine local heap address");

    
    if (H5B_iterate(oloc->file, H5B_SNODE, stab.btree_addr, H5G__node_sumup, num_objs) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_BADITER, FAIL, "iteration operator failed");

done:
    FUNC_LEAVE_NOAPI_TAG(ret_value)
} 

herr_t
H5G__stab_bh_size(H5F_t *f, const H5O_stab_t *stab, H5_ih_info_t *bh_info)
{
    hsize_t    snode_size; 
    H5B_info_t bt_info;    
    herr_t     ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(stab);
    assert(bh_info);

    
    snode_size = 0;

    
    if (H5B_get_info(f, H5B_SNODE, stab->btree_addr, &bt_info, H5G__node_iterate_size, &snode_size) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "iteration operator failed");

    
    bh_info->index_size += snode_size + bt_info.size;

    
    if (H5HL_heapsize(f, stab->heap_addr, &(bh_info->heap_size)) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "iteration operator failed");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5G__stab_get_name_by_idx_cb(const H5G_entry_t *ent, void *_udata)
{
    H5G_bt_it_gnbi_t *udata = (H5G_bt_it_gnbi_t *)_udata;
    size_t            name_off;            
    const char       *name;                
    size_t            block_size;          
    herr_t            ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(ent);
    assert(udata && udata->heap);

    
    block_size = H5HL_heap_get_size(udata->heap);

    
    name_off = ent->name_off;

    if ((name = (const char *)H5HL_offset_into(udata->heap, name_off)) == NULL)
        HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "unable to get symbol table link name");

    if (NULL == (udata->name = H5MM_strndup(name, (block_size - name_off))))
        HGOTO_ERROR(H5E_SYM, H5E_CANTCOPY, FAIL, "unable to duplicate symbol table link name");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5G__stab_get_name_by_idx(const H5O_loc_t *oloc, H5_iter_order_t order, hsize_t n, char *name,
                          size_t name_size, size_t *name_len)
{
    H5HL_t          *heap = NULL;           
    H5O_stab_t       stab;                  
    H5G_bt_it_gnbi_t udata;                 
    bool             udata_valid = false;   
    herr_t           ret_value   = SUCCEED; 

    
    memset(&udata, 0, sizeof(udata));

    FUNC_ENTER_PACKAGE

    
    assert(oloc);

    
    if (NULL == H5O_msg_read(oloc, H5O_STAB_ID, &stab))
        HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "unable to determine local heap address");

    
    if (NULL == (heap = H5HL_protect(oloc->file, stab.heap_addr, H5AC__READ_ONLY_FLAG)))
        HGOTO_ERROR(H5E_SYM, H5E_CANTPROTECT, FAIL, "unable to protect symbol table heap");

    
    if (order == H5_ITER_DEC) {
        hsize_t nlinks = 0; 

        
        if (H5B_iterate(oloc->file, H5B_SNODE, stab.btree_addr, H5G__node_sumup, &nlinks) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_BADITER, FAIL, "iteration operator failed");

        
        n = nlinks - (n + 1);
    } 

    
    udata.common.idx      = n;
    udata.common.num_objs = 0;
    udata.common.op       = H5G__stab_get_name_by_idx_cb;
    udata.heap            = heap;
    udata.name            = NULL;
    udata_valid           = true;

    
    if (H5B_iterate(oloc->file, H5B_SNODE, stab.btree_addr, H5G__node_by_idx, &udata) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_BADITER, FAIL, "iteration operator failed");

    
    if (udata.name == NULL)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "index out of bound");

    
    *name_len = strlen(udata.name);

    
    if (name) {
        strncpy(name, udata.name, MIN((*name_len + 1), name_size));
        if (*name_len >= name_size)
            name[name_size - 1] = '\0';
    } 

done:
    
    if (heap && H5HL_unprotect(heap) < 0)
        HDONE_ERROR(H5E_SYM, H5E_CANTUNPROTECT, FAIL, "unable to unprotect symbol table heap");

    
    if (udata_valid && udata.name != NULL)
        H5MM_xfree(udata.name);

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5G__stab_lookup_cb(const H5G_entry_t *ent, void *_udata)
{
    H5G_stab_fnd_ud_t *udata     = (H5G_stab_fnd_ud_t *)_udata; 
    herr_t             ret_value = SUCCEED;                     

    FUNC_ENTER_PACKAGE

    
    if (udata->lnk)
        
        if (H5G__ent_to_link(ent, udata->heap, udata->lnk) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_CANTCONVERT, FAIL, "unable to convert symbol table entry to link");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5G__stab_lookup(const H5O_loc_t *grp_oloc, const char *name, bool *found, H5O_link_t *lnk)
{
    H5HL_t           *heap = NULL;         
    H5G_bt_lkp_t      bt_udata;            
    H5G_stab_fnd_ud_t udata;               
    H5O_stab_t        stab;                
    herr_t            ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(grp_oloc && grp_oloc->file);
    assert(name && *name);
    assert(found);
    assert(lnk);

    
    if (NULL == H5O_msg_read(grp_oloc, H5O_STAB_ID, &stab))
        HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "can't read message");

    
    if (NULL == (heap = H5HL_protect(grp_oloc->file, stab.heap_addr, H5AC__READ_ONLY_FLAG)))
        HGOTO_ERROR(H5E_SYM, H5E_CANTPROTECT, FAIL, "unable to protect symbol table heap");

    
    udata.name = name;
    udata.lnk  = lnk;
    udata.heap = heap;

    
    bt_udata.common.name       = name;
    bt_udata.common.heap       = heap;
    bt_udata.common.block_size = H5HL_heap_get_size(heap);
    bt_udata.op                = H5G__stab_lookup_cb;
    bt_udata.op_data           = &udata;

    
    if (H5B_find(grp_oloc->file, H5B_SNODE, stab.btree_addr, found, &bt_udata) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_NOTFOUND, FAIL, "not found");

done:
    
    if (heap && H5HL_unprotect(heap) < 0)
        HDONE_ERROR(H5E_SYM, H5E_CANTUNPROTECT, FAIL, "unable to unprotect symbol table heap");

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5G__stab_lookup_by_idx_cb(const H5G_entry_t *ent, void *_udata)
{
    H5G_bt_it_lbi_t *udata     = (H5G_bt_it_lbi_t *)_udata;
    herr_t           ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(ent);
    assert(udata && udata->heap);

    
    if (H5G__ent_to_link(ent, udata->heap, udata->lnk) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTCONVERT, FAIL, "unable to convert symbol table entry to link");
    udata->found = true;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5G__stab_lookup_by_idx(const H5O_loc_t *grp_oloc, H5_iter_order_t order, hsize_t n, H5O_link_t *lnk)
{
    H5HL_t         *heap = NULL;         
    H5G_bt_it_lbi_t udata;               
    H5O_stab_t      stab;                
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(grp_oloc && grp_oloc->file);
    assert(lnk);

    
    if (NULL == H5O_msg_read(grp_oloc, H5O_STAB_ID, &stab))
        HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "unable to determine local heap address");

    
    if (NULL == (heap = H5HL_protect(grp_oloc->file, stab.heap_addr, H5AC__READ_ONLY_FLAG)))
        HGOTO_ERROR(H5E_SYM, H5E_CANTPROTECT, FAIL, "unable to protect symbol table heap");

    
    if (order == H5_ITER_DEC) {
        hsize_t nlinks = 0; 

        
        if (H5B_iterate(grp_oloc->file, H5B_SNODE, stab.btree_addr, H5G__node_sumup, &nlinks) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_BADITER, FAIL, "iteration operator failed");

        
        n = nlinks - (n + 1);
    } 

    
    udata.common.idx      = n;
    udata.common.num_objs = 0;
    udata.common.op       = H5G__stab_lookup_by_idx_cb;
    udata.heap            = heap;
    udata.lnk             = lnk;
    udata.found           = false;

    
    if (H5B_iterate(grp_oloc->file, H5B_SNODE, stab.btree_addr, H5G__node_by_idx, &udata) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_BADITER, FAIL, "iteration operator failed");

    
    if (!udata.found)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "index out of bound");

done:
    
    if (heap && H5HL_unprotect(heap) < 0)
        HDONE_ERROR(H5E_SYM, H5E_CANTUNPROTECT, FAIL, "unable to unprotect symbol table heap");

    FUNC_LEAVE_NOAPI(ret_value)
} 

#ifndef H5_STRICT_FORMAT_CHECKS

herr_t
H5G__stab_valid(H5O_loc_t *grp_oloc, H5O_stab_t *alt_stab)
{
    H5O_stab_t stab;                
    H5HL_t    *heap    = NULL;      
    bool       changed = false;     
    herr_t     bt_status;           
    herr_t     ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE_TAG(grp_oloc->addr)

    
    assert(grp_oloc);
    assert(alt_stab);

    
    if (NULL == H5O_msg_read(grp_oloc, H5O_STAB_ID, &stab))
        HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "unable to read symbol table message");

    
    H5E_PAUSE_ERRORS
        {
            bt_status = H5B_valid(grp_oloc->file, H5B_SNODE, stab.btree_addr);
        }
    H5E_RESUME_ERRORS

    if (bt_status < 0) {
        
        if (H5B_valid(grp_oloc->file, H5B_SNODE, alt_stab->btree_addr) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_BADVALUE, FAIL, "unable to locate b-tree");
        else {
            
            stab.btree_addr = alt_stab->btree_addr;
            changed         = true;
        } 
    }     

    
    H5E_PAUSE_ERRORS
        {
            heap = H5HL_protect(grp_oloc->file, stab.heap_addr, H5AC__READ_ONLY_FLAG);
        }
    H5E_RESUME_ERRORS

    if (NULL == heap) {
        
        if (NULL == (heap = H5HL_protect(grp_oloc->file, alt_stab->heap_addr, H5AC__READ_ONLY_FLAG)))
            HGOTO_ERROR(H5E_SYM, H5E_CANTPROTECT, FAIL, "unable to locate heap");
        else {
            
            stab.heap_addr = alt_stab->heap_addr;
            changed        = true;
        } 
    }     

    
    if (changed)
        if (H5O_msg_write(grp_oloc, H5O_STAB_ID, 0, H5O_UPDATE_TIME | H5O_UPDATE_FORCE, &stab) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_CANTSET, FAIL, "unable to correct symbol table message");

done:
    
    if (heap && H5HL_unprotect(heap) < 0)
        HDONE_ERROR(H5E_SYM, H5E_CANTUNPROTECT, FAIL, "unable to unprotect symbol table heap");

    FUNC_LEAVE_NOAPI_TAG(ret_value)
} 
#endif 
