% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gev_mle.R
\name{gev_mle}
\alias{gev_mle}
\title{GEV ML Inference with Adjustment for Missing Data}
\usage{
gev_mle(
  data,
  block_length,
  block,
  adjust = TRUE,
  discard = 0,
  init = "quartiles",
  ...
)
}
\arguments{
\item{data}{Either
\itemize{
\item a numeric vector containing a time series of raw data,
\item an object returned from \code{\link{block_maxima}}, a list with components
\code{maxima}, \code{notNA} and \code{n},
\item a data frame or named list containing the same information, that is, the
variables \code{maxima}, \code{notNA} and \code{n}, as an object returned from
\code{\link{block_maxima}}, such as the data frame \code{\link{BrestSurgeMaxima}}.
}}

\item{block_length}{A numeric scalar. Used calculate the maxima of disjoint
blocks of \code{block_length} contiguous values in the vector \code{data}.
If \code{length(data)} is not an integer multiple of \code{block_length} then
the values at the end of \code{data} that do not constitute a complete block
of length \code{block_length} are discarded, without warning.}

\item{block}{A numeric vector with the same length as \code{data}. The value of
\code{block[i]} indicates the block into which \code{data[i]} falls. For example,
\code{block} could provide the year in which observation \code{i} was observed.}

\item{adjust}{A logical scalar or a numeric scalar in \verb{[0, 100]}.
\itemize{
\item If \code{adjust = TRUE} then the adjustment, described in \strong{Details}, for the
numbers of non-missing values underlying each block maximum is performed.
\item If \code{adjust = FALSE} then no adjustment is made, that is, the block maxima
are treated as if the underlying raw data have no missing values.
}}

\item{discard}{A numeric scalar. Any block maximum for which greater than
\code{discard} percent of the underlying raw values were missing is discarded.
Whether or not an adjustment for missingness is made for the block maxima
that remain is determined by \code{adjust}.}

\item{init}{Either a character scalar, one of \code{"quartiles"} or \code{"moments"},
or a numeric vector of length 3 giving initial estimates of the GEV
location, scale and shape parameters: \eqn{\mu}, \eqn{\sigma} and
\eqn{\xi}. If \code{init = "quartiles"} then initial estimates of \eqn{\mu} and
\eqn{\sigma} are based on sample quartiles of block maxima, ignoring the
underlying numbers of non-missing raw data, and a value of 0 for
\eqn{\xi}. If \code{init = "moments"} then instead we use the sample mean and
variance of these maxima and an initial value of 0.1 for \eqn{\xi}.}

\item{...}{Further arguments to be passed to \code{\link[stats:optim]{stats::optim}}.}
}
\value{
A list, returned from \code{\link[stats:optim]{stats::optim}} (the MLEs are in the
component \code{par}), with the additional components:
\itemize{
\item \code{loglik}: value of the maximised log-likelihood.
\item \code{vcov}: estimated variance-covariance matrix of the parameters.
\item \code{se}: estimated standard errors of the parameters.
\item \code{maxima}: the vector of block maxima used to fit the model.
\item \code{notNA}: the number of non-missing raw values on which the maxima in
\code{maxima} are based.
\item \code{n}: the maximal block length, that is, the largest number of values that
could have been observed in each of these blocks.
\item \verb{adjust,discard} : the values of these input arguments.
}

The call to \code{gev_mle} is provided in the attribute \code{"call"}.

The class of the returned object is \code{c("evmissing", "mle", "list")}.

Objects inheriting from class \code{"evmissing"} have \code{coef}, \code{logLik}, \code{nobs},
\code{summary}, \code{vcov} and \code{confint} methods.  See \code{\link{evmissing_methods}}.
}
\description{
Fits a GEV distribution to block maxima using maximum likelihood estimation,
with the option to make an adjustment for the numbers of non-missing raw
values in each block. The GEV location and scale parameters are adjusted to
reflect the proportion of raw values that are missing.
}
\details{
If \code{data} is numeric vector then exactly one of the arguments
\code{block_length} or \code{block} must be supplied. The parameters are fitted
using maximum likelihood estimation.

The adjustment for the numbers of non-missing values underlying the block
maxima is based on the strong assumption that missing values occur
completely at random. We suppose that a block maximum \eqn{M_n} based on
a full (no missing raw values) block of length \eqn{n} has a
\eqn{\text{GEV}(\mu, \sigma, \xi)} distribution, with distribution function
\eqn{G(x)}. Let \eqn{n_i} be the number of non-missing values in block \eqn{i}
and let \eqn{M_{n_i}} denote the block maximum of such a block. We suppose
that \eqn{M_{n_i}} has a \eqn{\text{GEV}(\mu(n_i), \sigma(n_i), \xi)}
distribution, where \deqn{\mu(n_i) = \mu + \sigma [(n_i/n)^\xi -1] / \xi,}
\deqn{\sigma(n_i) = \sigma (n_i/n)^\xi.}

These expressions are based on inferring the parameters of an approximate
GEV distribution for \eqn{M_{n_i}} from its approximate distribution function
\eqn{[G(x)]^{n_i/n}}.

A likelihood is constructed as the product of contributions from the maxima
from distinct blocks, under the assumption that these maxima are
independent. Let \eqn{\theta = (\mu, \sigma, \xi)} and let
\eqn{\ell_F(\underline{z}; \theta)} denote the usual, unadjusted, GEV
log-likelihood for the full-data case where there are no missing values.
It can be shown that our adjusted log-likelihood
\eqn{\ell(\theta, \underline{z})} is given by

\deqn{\ell(\theta, \underline{z}) = \ell_F(\underline{z}; \theta) -
      \sum_{i=1}^n p_i \log G(z_i; \theta)}

where \eqn{p_i = 1 - n_i / n} is the proportion of missing values in block
\eqn{i}.

The negated log-likelihood is minimised using a call to
\code{\link[stats:optim]{stats::optim}} with \code{hessian = TRUE}. If \code{\link[stats:optim]{stats::optim}} throws an error
then a warning is produced and the returned object has \code{NA} values for
the components \code{par}, \code{loglik}, \code{vcov} and \code{se} and an extra component
\code{optim_error} containing the error message. If the estimated observed
information matrix is singular then a warning is produced and the returned
object has \code{NA} values for the components \code{vcov} and \code{se}.
}
\examples{
## Simulate raw data from an exponential distribution

set.seed(13032025)
blocks <- 50
block_length <- 365
sdata <- sim_data(blocks = blocks, block_length = block_length)

# sdata$data_full have no missing values
# sdata$data_miss have had missing values created artificially

# Fit a GEV distribution to block maxima from the full data
fit1 <- gev_mle(sdata$data_full, block_length = sdata$block_length)
summary(fit1)

# An identical fit supplying the block indicator instead of block_length
fit1b <- gev_mle(sdata$data_full, block = sdata$block)
summary(fit1b)

# Make adjustment for the numbers of non-missing values per block
fit2 <- gev_mle(sdata$data_miss, block_length = sdata$block_length)
summary(fit2)

# Do not make the adjustment
fit3 <- gev_mle(sdata$data_miss, block_length = sdata$block_length,
                adjust = FALSE)
summary(fit3)

# Remove all block maxima with greater than 25\% missing values and
# do not make the adjustment
fit4 <- gev_mle(sdata$data_miss, block_length = sdata$block_length,
                adjust = FALSE, discard = 25)
summary(fit4)

## Plymouth ozone data

# Calculate the values in Table 4 of Simpson and Northrop (2026)
# discard = 50 is chosen to discard data from 2001 and 2006
fit1 <- gev_mle(PlymouthOzoneMaxima)
fit2 <- gev_mle(PlymouthOzoneMaxima, adjust = FALSE)
fit3 <- gev_mle(PlymouthOzoneMaxima, discard = 50)
fit4 <- gev_mle(PlymouthOzoneMaxima, adjust = FALSE, discard = 50)
se <- function(x) return(sqrt(diag(vcov(x))))
MLEs <- cbind(coef(fit1), coef(fit2), coef(fit3), coef(fit4))
SEs <- cbind(se(fit1), se(fit2), se(fit3), se(fit4))
round(MLEs, 2)
round(SEs, 2)
}
