#' Permute DRI
#'
#' `permute_dri` tests whether the links between considerations and policy
#' preferences are consistent or likely due to chance
#'
#' @param data raw DRI survey dataframe
#' @param iterations number permutations to generate
#' @param verbose a logical flag to print time of permutation
#' @param summary a logical indicating whether to return the raw data or summary
#' of test results; raw data is optimal for plotting permutation results
#'
#' @import dplyr
#'
#' @returns dataframe with permutation test results, raw or summarized.
#' Summarized results include the number of participants, `n`, the observed DRI,
#' `obs_dri`, the number of permutations conducted, `n_perm`, the mean
#' permutation DRI, `mean_perm_dri`, and the frequency which the permutation DRI
#' is greater or equal to the observed DRI, `p`
#' @export
#'
#' @examples
#'
#' # get pre-deliberation (stage_id == 1) data from Zukunft case
#' data <- human_data[human_data$stage_id == 1 & human_data$case == "Zukunft", ]
#'
#' # permute DRI 100 times
#' permute_dri(data, iterations = 100)
#'
permute_dri <- function(data, iterations = 10000, verbose = FALSE, summary = TRUE) {

  res <- list()

  # GET OBSERVED DRI
  ic <- get_dri_ic(data)
  obs_dri <- get_dri(ic)

  raw <- tibble(
    dri = obs_dri,
    source = "observed",
  )

  # RUN PERMUTATION
  # get preferences
  pref_cols <- grep("^P\\d", names(data), value = TRUE)
  shuffled_data <- data
  dri_shuffle <- list()

  time_start <- Sys.time()

  # permutation loop
  for (i in 1:iterations) {

    # shuffle preferences
    shuffled_data[pref_cols] <-
      shuffled_data[sample(1:nrow(shuffled_data)), pref_cols]

    # get DRI
    ic <- get_dri_ic(shuffled_data)
    dri <- get_dri(ic)

    dri_shuffle[[length(dri_shuffle) + 1]] <- tibble(
      dri,
      source = "permutation",
      iteration = i
    )

  }

  time_end <- Sys.time()

  elapsed_time <- format(difftime(time_end, time_start, units = "auto"), digits = 3)

  if (verbose) cat(iterations, "permutations completed in", elapsed_time, "\n")

  # merge permutation data
  dri_shuffle <- bind_rows(dri_shuffle)

  raw <- bind_rows(raw, dri_shuffle)

  # calculate p
  p <- nrow(dri_shuffle %>% filter(dri >= obs_dri)) / nrow(dri_shuffle)

  #
  res <- tibble(
    n = nrow(data),
    obs_dri = obs_dri,
    n_perm = nrow(dri_shuffle),
    mean_perm_dri = mean(dri_shuffle$dri, na.rm = TRUE),
    p = p,
  )

  if (summary) return(res) else return (raw)

}
