% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/prepare_trait_data.R
\name{select_best_trait_model_from_geiger}
\alias{select_best_trait_model_from_geiger}
\title{Compare trait evolutionary model fits with AICc and Akaike's weights}
\usage{
select_best_trait_model_from_geiger(list_model_fits)
}
\arguments{
\item{list_model_fits}{Named list with the results of a model fit with \code{\link[geiger:fitContinuous]{geiger::fitContinuous()}} or \code{\link[geiger:fitDiscrete]{geiger::fitDiscrete()}} in each element.}
}
\value{
The function returns a list with three elements.
\itemize{
\item \verb{$model_comparison_df} Data.frame summarizing information to compare model fits. It includes the model name (\verb{$model}),
the log-likelihood (\verb{$logLik}), the number of free-parameters (\verb{$k}), the AIC (\verb{$AICc}), the corrected AIC (\verb{$AICc}),
the delta to the best/lowest AICc (\verb{$delta_AICc}), the Akaike weights (\verb{$Akaike_weights}), and their rank based on AICc (\verb{$rank}).
\item \verb{$best_model_name} Character string. Name of the best model.
\item \verb{$best_model_fit} List containing the output of \code{\link[geiger:fitContinuous]{geiger::fitContinuous()}} or \code{\link[geiger:fitDiscrete]{geiger::fitDiscrete()}} for the model with the best fit.
}
}
\description{
Compare evolutionary models fit with \code{\link[geiger:fitContinuous]{geiger::fitContinuous()}} or \code{\link[geiger:fitDiscrete]{geiger::fitDiscrete()}}
using AICc and Akaike's weights. Generate a data.frame summarizing information.
Identify the best model and extract its results.
}
\examples{

# ----- Example 1: Continuous data ----- #

# Load phylogeny and tip data
library(phytools)
data(eel.tree)
data(eel.data)

# Extract body size
eel_data <- stats::setNames(eel.data$Max_TL_cm,
                            rownames(eel.data))

\donttest{ # (May take several minutes to run)
# Fit BM model
BM_fit <- geiger::fitContinuous(phy = eel.tree, dat = eel_data, model = "BM")
# Fit EB model
EB_fit <- geiger::fitContinuous(phy = eel.tree, dat = eel_data, model = "EB")
# Fit OU model
OU_fit <- geiger::fitContinuous(phy = eel.tree, dat = eel_data, model = "OU")

# Store models
list_model_fits <- list(BM = BM_fit, EB = EB_fit, OU = OU_fit)

# Compare models
model_comparison_output <- select_best_trait_model_from_geiger(list_model_fits = list_model_fits)

# Explore output
str(model_comparison_output, max.level = 2)

# Print comparison
print(model_comparison_output$models_comparison_df)

# Print best model fit
print(model_comparison_output$best_model_fit) }

# ----- Example 2: Categorical data ----- #

# Load phylogeny and tip data
library(phytools)
data(eel.tree)
data(eel.data)

# Extract feeding mode data
eel_data <- stats::setNames(eel.data$feed_mode, rownames(eel.data))
table(eel_data)

\donttest{ # (May take several minutes to run)
# Fit ER model
ER_fit <- geiger::fitDiscrete(phy = eel.tree, dat = eel_data, model = "ER")
# Fit ARD model
ARD_fit <- geiger::fitDiscrete(phy = eel.tree, dat = eel_data, model = "ARD")

# Store models
list_model_fits <- list(ER = ER_fit, ARD = ARD_fit)

# Compare models
model_comparison_output <- select_best_trait_model_from_geiger(list_model_fits = list_model_fits)

# Explore output
str(model_comparison_output, max.level = 2)

# Print comparison
print(model_comparison_output$models_comparison_df)

# Print best model fit
print(model_comparison_output$best_model_fit) }

}
\seealso{
\code{\link[geiger:fitContinuous]{geiger::fitContinuous()}} \code{\link[geiger:fitDiscrete]{geiger::fitDiscrete()}}
}
\author{
Maël Doré
}
