% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/musycModel.R
\name{evalMusycModel}
\alias{evalMusycModel}
\title{Evaluate MuSyC Response Surfaces}
\usage{
evalMusycModel(DA, DB, mupar, calcderivs = FALSE)
}
\arguments{
\item{DA}{A vector of concentrations of drug A in a combination (values 0
and \code{Inf} are permitted). Must be length 1 or the same length as \code{DB}.}

\item{DB}{A vector of concentrations of drug B in a combination (values 0
and \code{Inf} are permitted). Must be length 1 or the same length as \code{DA}.}

\item{mupar}{A MuSyC response surface parameter vector; may be length 8, 10,
or 12 (see details for specifics of MuSyC parameters)}

\item{calcderivs}{Primarily used by fitting functions for non-linear
optimization.  If \code{FALSE} (the default), the function returns a vector of
response values; if \code{TRUE}, it returns a list including the partial
derivatives of the BRAID parameters.}
}
\value{
If \code{calcderivs} is \code{FALSE}, a numeric vector the same length as \code{DA}
and/or \code{DB} with the predicted MuSyC response surface values.  If
\code{calcderivs} is \code{TRUE}, a list with two elements: \code{value}, containing the
response surface values, and \code{derivatives}, a matrix with as many rows as
\code{value} has elements, and all columns containing the partial derivatives of
the response surface with respect to the fitted MuSyC response surface
parameters
}
\description{
Evaluates the Mulitdimensional Synergy of Combinations (MuSyC) model of
combined action for the given values and parameters (Wooten \emph{et al.} 2021).
}
\details{
The multi-dimensional synergy of combinatoins, or MySyC, model is a
parametric response surface model introduced by Wooten et al. in 2021.  The
method models the effect of combination by simulating occupancy in four
compartments in which compounds are affected or unaffected by either drug.
The full MuSyC model can be specified by a total of twelve parameters:
\itemize{
\item \code{IDMA}: dose of median effect of first drug
\item \code{IDMB}: dose of median effect of second drug
\item \code{na}: Hill slope of first drug
\item \code{nb}: Hill slope of second drug
\item \code{alpha12}: factor by which first drug potentiates the second
\item \code{alpha21}: factor by which second drug potentiates the first
\item \code{gamma12}: factor by which first drug increases second drug's Hill slope
\item \code{gamma21}: factor by which second drug increases first drug's Hill slope
\item \code{E0} - the observed effect when unaffected by either drug
\item \code{EfA} - the observed effect when affected by drug 1 but not drug 2
\item \code{EfB} - the observed effect when affected by drug 2 but not drug 1
\item \code{Ef} - the observed effect when affected by both drugs
}

In practice, \code{gamma12} and \code{gamma21} are rarely used, so a ten-element
parameter vector specifies the other 10 values and assumes that \code{gamma12}
and \code{gamma21} are both equal to 1.  In some cases it is even useful to
specify a MuSyC surface with no interaction at all with an eight-element
vector, in which case \code{alpha12}, \code{alpha21}, \code{gamma12}, and \code{gamma21} are all
set equal to 1.
}
\examples{
efficacyPar <- c(
    1, 1, 3, 3,
                      # Omitted shape synergy parameters assume to be 1
    0, 100, 100, 125  # Elevated Ef indicates efficacy synergy
)
potencyPar <- c(
    1, 1, 3, 3,
    10, 15,           # alphas above 1 indicate potency synergy
    0, 100, 100, 100  # No efficacy synergy
)

concentrations <- c(0, 2^(-3:3))
surface <- data.frame(
    concA = rep(concentrations,each=length(concentrations)),
    concB = rep(concentrations,times=length(concentrations))
)
surface$efficacy <- evalMusycModel(surface$concA, surface$concB, efficacyPar)
surface$potency  <- evalMusycModel(surface$concA, surface$concB, potencyPar)

head(surface)
}
\references{
Wooten, David J, Christian T Meyer, Alexander LR Lubbock, Vito Quaranta,
and Carlos F Lopez. 2021. “MuSyC Is a Consensus Framework That Unifies
Multi-Drug Synergy Metrics for Combinatorial Drug Discovery.”
\emph{Nature Communications} \strong{12} (1): 4607.
}
