% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/orsf_control.R
\name{orsf_control}
\alias{orsf_control}
\alias{orsf_control_classification}
\alias{orsf_control_regression}
\alias{orsf_control_survival}
\title{Oblique random forest control}
\usage{
orsf_control(
  tree_type,
  method,
  scale_x,
  ties,
  net_mix,
  target_df,
  max_iter,
  epsilon,
  ...
)

orsf_control_classification(
  method = "glm",
  scale_x = TRUE,
  net_mix = 0.5,
  target_df = NULL,
  max_iter = 20,
  epsilon = 1e-09,
  ...
)

orsf_control_regression(
  method = "glm",
  scale_x = TRUE,
  net_mix = 0.5,
  target_df = NULL,
  max_iter = 20,
  epsilon = 1e-09,
  ...
)

orsf_control_survival(
  method = "glm",
  scale_x = TRUE,
  ties = "efron",
  net_mix = 0.5,
  target_df = NULL,
  max_iter = 20,
  epsilon = 1e-09,
  ...
)
}
\arguments{
\item{tree_type}{(\emph{character}) the type of tree. Valid options are
\itemize{
\item "classification", i.e., categorical outcomes
\item "regression", i.e., continuous outcomes
\item "survival", i.e., time-to event outcomes
}}

\item{method}{(\emph{character} or \emph{function}) how to identify linear
linear combinations of predictors. If \code{method} is a character value,
it must be one of:
\itemize{
\item 'glm': linear, logistic, and cox regression
\item 'net': same as 'glm' but with penalty terms
\item 'random': random draw from uniform distribution
}

If \code{method} is a \emph{function}, it will be used to identify  linear
combinations of predictor variables. \code{method} must in this case accept
three inputs named \code{x_node}, \code{y_node} and \code{w_node}, and should expect
the following types and dimensions:
\itemize{
\item \code{x_node} (\emph{matrix}; \code{n} rows, \code{p} columns)
\item \code{y_node} (\emph{matrix}; \code{n} rows, \code{2} columns)
\item \code{w_node} (\emph{matrix}; \code{n} rows, \code{1} column)
}

In addition, \code{method} must return a matrix with p rows and 1 column.}

\item{scale_x}{(\emph{logical}) if \code{TRUE}, values of predictors will be
scaled prior to each instance of finding a linear combination of
predictors, using summary values from the data in the current node
of the decision tree.}

\item{ties}{(\emph{character}) a character string specifying the method
for tie handling. Only relevant when modeling survival outcomes
and using a method that engages with tied outcome times.
If there are no ties, all the methods are equivalent. Valid options
are 'breslow' and 'efron'. The Efron approximation is the default
because it is more accurate when dealing with tied event times and
has similar computational efficiency compared to the Breslow method.}

\item{net_mix}{(\emph{double}) The elastic net mixing parameter. A value of 1
gives the lasso penalty, and a value of 0 gives the ridge penalty. If
multiple values of alpha are given, then a penalized model is fit using
each alpha value prior to splitting a node.}

\item{target_df}{(\emph{integer}) Preferred number of variables used in each
linear combination. For example, with \code{mtry} of 5 and \code{target_df} of 3,
we sample 5 predictors and look for the best linear combination using
3 of them.}

\item{max_iter}{(\emph{integer}) iteration continues until convergence
(see \code{eps} above) or the number of attempted iterations is equal to
\code{iter_max}.}

\item{epsilon}{(\emph{double}) When using most modeling based method,
iteration continues in the algorithm until the relative change in
some kind of objective is less than \code{epsilon}, or the absolute
change is less than \code{sqrt(epsilon)}.}

\item{...}{Further arguments passed to or from other methods (not currently used).}
}
\value{
an object of class \code{'orsf_control'}, which should be used as
an input for the \code{control} argument of \link{orsf}. Components are:
\itemize{
\item \code{tree_type}: type of trees to fit
\item \code{lincomb_type}: method for linear combinations
\item \code{lincomb_eps}: epsilon for convergence
\item \code{lincomb_iter_max}: max iterations
\item \code{lincomb_scale}: to scale or not.
\item \code{lincomb_alpha}: mixing parameter
\item \code{lincomb_df_target}: target degrees of freedom
\item \code{lincomb_ties_method}: method for ties in survival time
\item \code{lincomb_R_function}: R function for custom splits
}
}
\description{
Oblique random forest control
}
\details{
Adjust \code{scale_x} \emph{at your own risk}. Setting \code{scale_x = FALSE} will
reduce computation time but will also make the \code{orsf} model dependent
on the scale of your data, which is why the default value is \code{TRUE}.
}
\section{Examples}{
First we load some relevant packages

\if{html}{\out{<div class="sourceCode r">}}\preformatted{set.seed(329730)
suppressPackageStartupMessages(\{
 library(aorsf)
 library(survival)
 library(ranger)
 library(riskRegression)
\})
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{## Warning: package 'riskRegression' was built under R version 4.5.2
}\if{html}{\out{</div>}}
\subsection{Accelerated linear combinations}{

The accelerated ORSF ensemble is the default because it has a nice
balance of computational speed and prediction accuracy. It runs a single
iteration of Newton Raphson scoring on the Cox partial likelihood
function to find linear combinations of predictors.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{fit_accel <- orsf(pbc_orsf, 
                  control = orsf_control_survival(),
                  formula = Surv(time, status) ~ . - id,
                  tree_seeds = 329)
}\if{html}{\out{</div>}}
}

\subsection{Linear combinations with Cox regression}{

Setting inputs in \code{orsf_control_survival} to scale the X matrix and
repeat iterations until convergence allows you to run Cox regression in
each non-terminal node of each survival tree, using the regression
coefficients to create linear combinations of predictors:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{control_cph <- orsf_control_survival(method = 'glm', 
                                     scale_x = TRUE, 
                                     max_iter = 20)

fit_cph <- orsf(pbc_orsf, 
                control = control_cph,
                formula = Surv(time, status) ~ . - id,
                tree_seeds = 329)
}\if{html}{\out{</div>}}
}

\subsection{Linear combinations with penalized cox regression}{

Setting \code{method == 'net'} runs penalized Cox regression in each
non-terminal node of each survival tree. This can be really helpful if
you want to do feature selection within the node, but it is a lot slower
than the \code{'glm'} option.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{# select 3 predictors out of 5 to be used in
# each linear combination of predictors.

control_net <- orsf_control_survival(method = 'net', target_df = 3)

fit_net <- orsf(pbc_orsf, 
                control = control_net,
                formula = Surv(time, status) ~ . - id,
                tree_seeds = 329)
}\if{html}{\out{</div>}}
}

\subsection{Linear combinations with your own function}{

In addition to the built-in methods, customized functions can be used to
identify linear combinations of predictors. We’ll demonstrate a few
here.
\itemize{
\item The first uses random coefficients
}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{f_rando <- function(x_node, y_node, w_node)\{
 matrix(runif(ncol(x_node)), ncol=1) 
\}
}\if{html}{\out{</div>}}
\itemize{
\item The second derives coefficients from principal component analysis
}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{f_pca <- function(x_node, y_node, w_node) \{ 
 
 # estimate two principal components.
 pca <- stats::prcomp(x_node, rank. = 2)
 # use the second principal component to split the node
 pca$rotation[, 1L, drop = FALSE]
 
\}
}\if{html}{\out{</div>}}
\itemize{
\item The third uses \code{ranger()} inside of \code{orsf()}. This approach is very
similar to a method known as reinforcement learning trees (see the
\code{RLT} package), although our method of “muting” is very crude compared
to the method proposed by Zhu et al. 
}

\if{html}{\out{<div class="sourceCode r">}}\preformatted{f_rlt <- function(x_node, y_node, w_node)\{
 
 colnames(y_node) <- c('time', 'status')
 colnames(x_node) <- paste("x", seq(ncol(x_node)), sep = '')
 
 data <- as.data.frame(cbind(y_node, x_node))
 
 if(nrow(data) <= 10) 
  return(matrix(runif(ncol(x_node)), ncol = 1))
 
 fit <- ranger::ranger(data = data, 
                       formula = Surv(time, status) ~ ., 
                       num.trees = 25, 
                       num.threads = 1,
                       min.node.size = 5,
                       importance = 'permutation')
 
 out <- sort(fit$variable.importance, decreasing = TRUE)
 
 # "mute" the least two important variables
 n_vars <- length(out)
 if(n_vars > 4)\{
   out[c(n_vars, n_vars-1)] <- 0
 \}
 
 # ensure out has same variable order as input
 out <- out[colnames(x_node)]
 
 # protect yourself
 out[is.na(out)] <- 0
 
 matrix(out, ncol = 1)
 
\}
}\if{html}{\out{</div>}}

We can plug these functions into \code{orsf_control_custom()}, and then pass
the result into \code{orsf()}:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{fit_rando <- orsf(pbc_orsf,
                  Surv(time, status) ~ . - id,
                  control = orsf_control_survival(method = f_rando),
                  tree_seeds = 329)

fit_pca <- orsf(pbc_orsf,
                Surv(time, status) ~ . - id,
                control = orsf_control_survival(method = f_pca),
                tree_seeds = 329)

fit_rlt <- orsf(pbc_orsf, time + status ~ . - id, 
                control = orsf_control_survival(method = f_rlt),
                tree_seeds = 329)
}\if{html}{\out{</div>}}

So which fit seems to work best in this example? Let’s find out by
evaluating the out-of-bag survival predictions.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{risk_preds <- list(
 accel = fit_accel$pred_oobag,
 cph   = fit_cph$pred_oobag,
 net   = fit_net$pred_oobag,
 rando = fit_rando$pred_oobag,
 pca   = fit_pca$pred_oobag,
 rlt   = fit_rlt$pred_oobag
)

sc <- Score(object = risk_preds, 
            formula = Surv(time, status) ~ 1, 
            data = pbc_orsf, 
            summary = 'IPA',
            times = fit_accel$pred_horizon)
}\if{html}{\out{</div>}}

The AUC values, from highest to lowest:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{sc$AUC$score[order(-AUC)]
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{##     model times       AUC         se     lower     upper
##    <fctr> <num>     <num>      <num>     <num>     <num>
## 1:    net  1788 0.9151649 0.02025057 0.8754745 0.9548553
## 2:  accel  1788 0.9095628 0.02143250 0.8675558 0.9515697
## 3:    cph  1788 0.9095628 0.02143250 0.8675558 0.9515697
## 4:    rlt  1788 0.9089871 0.02099354 0.8678406 0.9501337
## 5:  rando  1788 0.9062197 0.02148854 0.8641029 0.9483365
## 6:    pca  1788 0.8999479 0.02226683 0.8563057 0.9435901
}\if{html}{\out{</div>}}

And the indices of prediction accuracy:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{sc$Brier$score[order(-IPA), .(model, times, IPA)]
}\if{html}{\out{</div>}}

\if{html}{\out{<div class="sourceCode">}}\preformatted{##         model times   IPA
##        <fctr> <num> <num>
## 1: Null model  1788    NA
## 2:      accel  1788    NA
## 3:        cph  1788    NA
## 4:        net  1788    NA
## 5:      rando  1788    NA
## 6:        pca  1788    NA
## 7:        rlt  1788    NA
}\if{html}{\out{</div>}}

From inspection,
\itemize{
\item \code{net}, \code{accel}, and \code{rlt} have high discrimination and index of
prediction accuracy.
\item \code{rando} and \code{pca} do less well, but they aren’t bad.
}
}
}

\seealso{
linear combination control functions
\code{\link{orsf_control_cph}()},
\code{\link{orsf_control_custom}()},
\code{\link{orsf_control_fast}()},
\code{\link{orsf_control_net}()}
}
\concept{orsf_control}
