
#' @title Relative Area Under the Disease Progress Curve (rAUDPC)
#'
#' @description
#' Calculates the Relative Area Under the Disease Progress Curve (rAUDPC),
#' which expresses AUDPC as a percentage of the maximum possible disease
#' development over the observation period.
#'
#' @param time A numeric vector of time points (e.g., days after sowing or inoculation).
#' Values must be in strictly increasing order.
#'
#' @param severity A numeric vector of disease severity values
#' corresponding to each time point.
#'
#' @param max_severity A numeric value representing the maximum possible
#' disease severity on the rating scale.
#'
#' @return A numeric value representing relative AUDPC (percentage).
#'
#' @details
#' rAUDPC is calculated as:
#' \deqn{
#' rAUDPC = \left( \frac{AUDPC}{(T_{max} - T_{min}) \times Y_{max}} \right) \times 100
#' }
#'
#' where \eqn{AUDPC} is the area under the disease progress curve,
#' \eqn{T_{max}} and \eqn{T_{min}} are the maximum and minimum time points,
#' and \eqn{Y_{max}} is the maximum disease severity.
#'
#' @export
#'
#' @examples {
#' time <- c(0, 7, 14, 21, 28)
#' severity <- c(5, 15, 30, 45, 60)
#' max_severity <- 100
#' rAUDPC(time, severity, max_severity)
#' }
rAUDPC <- function(time, severity, max_severity) {

  # Input checks
  if (length(time) != length(severity)) {
    stop("Time and severity must be of equal length")
  }

  if (any(diff(time) <= 0)) {
    stop("Time must be in increasing order")
  }

  # AUDPC calculation (trapezoidal rule)
  audpc <- sum(
    ((severity[-length(severity)] + severity[-1]) / 2) *
      (time[-1] - time[-length(time)])
  )

  # Relative AUDPC (%)
  raudpc <- (audpc / ((max(time) - min(time)) * max_severity)) * 100

  return(round(raudpc, 2))
}
