# ChenExp.R
#' Chen-Exponential Distribution
#'
#' Provides density, distribution, quantile, random generation,
#' and hazard functions for the Chen-Exponential distribution.
#'
#' The Chen-Exponential distribution is parameterized by the parameters
#' \eqn{\alpha > 0}, \eqn{\beta > 0}, and \eqn{\lambda > 0}.
#'
#' @details
#' The Chen-Exponential distribution has CDF:
#'
#' \deqn{
#' F(x;\,\alpha,\beta,\lambda) = \, 1-\exp \left\{\lambda\left[1-\exp 
#' \left\{\left(e^{\beta x}-1\right)^\alpha\right\} \right] \right\}, \quad x > 0.
#' }
#'
#' where \eqn{\alpha}, \eqn{\beta}, and \eqn{\lambda} are the parameters.
#'
#' The following functions are included:
#' \itemize{
#'   \item \code{dchen.exp()} — Density function
#'   \item \code{pchen.exp()} — Distribution function
#'   \item \code{qchen.exp()} — Quantile function
#'   \item \code{rchen.exp()} — Random generation
#'   \item \code{hchen.exp()} — Hazard function
#' }
#'
#' @usage
#' dchen.exp(x, alpha, beta, lambda, log = FALSE)
#' pchen.exp(q, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
#' qchen.exp(p, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
#' rchen.exp(n, alpha, beta, lambda)
#' hchen.exp(x, alpha, beta, lambda)
#'
#' @param x,q numeric vector of quantiles (x, q)
#' @param p numeric vector of probabilities (0 < p < 1)
#' @param n number of observations (integer > 0)
#' @param alpha positive numeric parameter
#' @param beta positive numeric parameter
#' @param lambda positive numeric parameter
#' @param log logical; if TRUE, returns log-density
#' @param lower.tail logical; if TRUE (default), probabilities are 
#' \eqn{P[X \le x]} otherwise, \eqn{P[X > x]}.
#' @param log.p logical; if TRUE, probabilities are given as log(p)
#'
#' @return
#' \itemize{
#'   \item \code{dchen.exp}: numeric vector of (log-)densities
#'   \item \code{pchen.exp}: numeric vector of probabilities
#'   \item \code{qchen.exp}: numeric vector of quantiles
#'   \item \code{rchen.exp}: numeric vector of random variates
#'   \item \code{hchen.exp}: numeric vector of hazard values
#' }
#'
#' @references
#' Chen, Z. (2000). A new two-parameter lifetime distribution with bathtub shape 
#' or increasing failure rate function. \emph{Statistics & Probability Letters}, 
#' \bold{49}, 155--161.  
#'
#' Sapkota, L.P., & Kumar, V. (2023). 
#' Chen Exponential Distribution with Applications to Engineering Data.
#' \emph{International Journal of Statistics and Reliability Engineering},
#' \bold{10(1)}, 33--47. 
#' 
#' Sapkota, L.P., Alsahangiti, A.M., Kumar, V. Gemeay, A.M., Bakr, M.E., Balogun, 
#' O.S., & Muse, A.H. (2023). Arc-Tangent Exponential Distribution With Applications
#' to Weather and Chemical Data Under Classical and Bayesian Approach, 
#' \emph{IEEE Access}, \bold{11}, 115462--115476. \doi{10.1109/ACCESS.2023.3324293}
#'
#' @examples
#' x <- seq(0.1, 1, 0.1)
#' dchen.exp(x, 1.5, 0.8, 2)
#' pchen.exp(x, 1.5, 0.8, 2)
#' qchen.exp(0.5, 1.5, 0.8, 2)
#' rchen.exp(10, 1.5, 0.8, 2)
#' hchen.exp(x, 1.5, 0.8, 2)
#' #Data
#' x <- stress
#' #ML Estimates    
#' params = list(alpha=2.5462, beta=0.0537, lambda=87.6028)
#' #P–P (probability–probability) plot
#' pp.plot(x, params = params, pfun = pchen.exp, fit.line=TRUE)
#'
#' #Q-Q (quantile–quantile) plot 
#' qq.plot(x, params = params, qfun = qchen.exp, fit.line=TRUE)
#'
#' # Goodness-of-Fit(GoF) and Model Diagnostics 
#' # Display plot; numerical summary stored in 'out'
#' out <- gofic(x, params = params, dfun = dchen.exp, 
#'              pfun = pchen.exp, plot=TRUE)
#' print.gofic(out)
#' @name ChenExp
#' @export
NULL

#' @rdname ChenExp
#' @usage NULL
#' @export 
dchen.exp <- function (x, alpha, beta, lambda, log = FALSE)
{
	if (!all(sapply(list(x, alpha, beta, lambda), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, beta, lambda) <= 0)) {
		stop("Parameters 'alpha', 'beta' and 'lambda' must be positive.")
	}

	pdf <- numeric(length(x))

	valid <- x > 0
	if (any(valid)) {		
	w <- exp(beta * x[valid])	
	u <- w - 1.0
	z <- exp(u ^ alpha)	
    pdf[valid] <- alpha * beta * lambda * w * (u ^(alpha-1)) * z * exp(lambda *(1-z))
    }
	if (log) 
       pdf <- log(pdf)
    return(pdf)   
}

#' @rdname ChenExp
#' @usage NULL
#' @export 
pchen.exp <- function(q, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(q, alpha, beta, lambda), is.numeric))) {
		stop("All arguments must be numeric.")
	}

	if (any(c(alpha, beta, lambda) <= 0)) {
		stop("Parameters 'alpha', 'beta' and 'lambda' must be positive.")
	}

	cdf <- numeric(length(q))

	valid <- q > 0
	if (any(valid)) {		
	z <- 1.0 - exp((exp(beta * q[valid]) -1.0)^alpha)
    cdf[valid] <- 1.0 - exp(lambda * z) 
    }
	if(!lower.tail)
        cdf <- 1.0 - cdf
    if(log.p)
        cdf <- log(cdf)
    return(cdf)
}

#' @rdname ChenExp
#' @usage NULL
#' @export
qchen.exp <- function(p, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
{
	if (!all(sapply(list(p, alpha, beta, lambda), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, beta, lambda) <= 0))
		stop("Invalid parameters: alpha, 'beta' and lambda must be positive.")
	
	if (log.p) p <- exp(p)
    if (!lower.tail) p <- 1 - p

    if (any(p < 0 | p > 1))
		stop("Probabilities 'p' must lie in [0, 1].")

	qtl <- numeric(length(p))
	
	is_zero <- p == 0
	is_one  <- p == 1
	valid  <- p > 0 & p < 1

	if (any(is_zero)) qtl[is_zero] <- 0
	if (any(is_one))  qtl[is_one]  <- Inf
	
	if (any(valid)) {	
    qtl[valid] <- (1/beta) * log(((log(1-(1/lambda)*log(1-p[valid]))) ^ (1/alpha)) +1.0) 
    }
    return(qtl)   
} 

#' @rdname ChenExp
#' @usage NULL
#' @export 
rchen.exp <- function(n, alpha, beta, lambda)
{
	if (!is.numeric(n) || length(n) != 1L || n <= 0 || n %% 1 != 0)
		stop("'n' must be a positive integer.") 
	if (!all(sapply(list(alpha, beta, lambda), is.numeric)))
		stop("'alpha', and 'lambda' must be numeric.")
	if (any(c(alpha, beta, lambda) <= 0))
		stop("Invalid parameters: alpha, 'beta', and lambda must be positive.")
  
    u <- runif(n)
    q <- qchen.exp(u, alpha, beta, lambda)
    return(q)
}

#' @rdname ChenExp
#' @usage NULL
#' @export 
hchen.exp <- function (x, alpha, beta, lambda)
{
	if (!all(sapply(list(x, alpha, beta, lambda), is.numeric)))
		stop("All arguments must be numeric.")
	if (any(c(alpha, beta, lambda, x) <= 0)) 
		stop("All parameters and 'x' must be positive.")  
    nume <- dchen.exp(x, alpha, beta, lambda, log=FALSE)
	surv <- 1.0 - pchen.exp(x, alpha, beta, lambda, lower.tail = TRUE, log.p = FALSE)
	hrf <- ifelse(surv > .Machine$double.eps, nume / surv, Inf)
    return(hrf)    
}
