% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/FCP_TPA.R
\name{FCP_TPA}
\alias{FCP_TPA}
\title{The functional CP-TPA algorithm}
\usage{
FCP_TPA(
  X,
  K,
  penMat,
  alphaRange,
  verbose = FALSE,
  tol = 1e-04,
  maxIter = 15,
  adaptTol = TRUE
)
}
\arguments{
\item{X}{The data tensor of dimensions \code{N x S1 x S2}.}

\item{K}{The number of eigentensors to be calculated.}

\item{penMat}{A list with entries \code{v} and \code{w}, containing a 
roughness penalty matrix for each direction of the image. The algorithm 
does not induce smoothness along observations (see Details).}

\item{alphaRange}{A list of length 2 with entries \code{v} and \code{w} , 
containing the range of smoothness parameters to test for each direction.}

\item{verbose}{Logical. If \code{TRUE}, computational details are given on 
the standard output during calculation of the FCP_TPA.}

\item{tol}{A numeric value, giving the tolerance for relative error values in
the algorithm. Defaults to \code{1e-4}. It is automatically multiplied by 
10 after \code{maxIter} steps, if \code{adaptTol = TRUE}.}

\item{maxIter}{A numeric value, the maximal iteration steps. Can be doubled, 
if \code{adaptTol = TRUE}.}

\item{adaptTol}{Logical. If \code{TRUE}, the tolerance is adapted (multiplied
by 10), if the algorithm has not converged after \code{maxIter} steps and 
another \code{maxIter} steps are allowed with the increased tolerance, see 
Details. Use with caution. Defaults to \code{TRUE}.}
}
\value{
\item{d}{A vector of length \code{K}, containing the numeric weights 
  \eqn{d_k} in the CP model.} \item{U}{A matrix of dimensions \code{N x K}, 
  containing the eigenvectors \eqn{u_k} in the first dimension.} \item{V}{A 
  matrix of dimensions \code{S1 x K}, containing the eigenvectors \eqn{v_k} 
  in the second dimension.} \item{W}{A matrix of dimensions \code{S2 x K}, 
  containing the eigenvectors \eqn{w_k} in the third dimension.}
}
\description{
This function implements the functional CP-TPA (FCP-TPA) algorithm, that 
calculates a smooth PCA for 3D tensor data (i.e. \code{N} observations of 2D 
images with dimension \code{S1 x S2}). The results are given in a 
CANDECOMP/PARAFRAC (CP) model format \deqn{X = \sum_{k = 1}^K d_k \cdot u_k 
\circ v_k \circ w_k}{X = \sum d_k u_k \%o\% v_k \%o\% w_k}  where 
\eqn{\circ}{\%o\%} stands for the outer product, \eqn{d_k} is a scalar and 
\eqn{u_k, v_k, w_k} are eigenvectors for each direction of the tensor. In 
this representation, the outer product \eqn{v_k \circ w_k}{v_k \%o\% w_k} can
be regarded as the \eqn{k}-th eigenimage, while \eqn{d_k \cdot u_k}{d_k u_k} 
represents the vector of individual scores for this eigenimage and each 
observation.
}
\details{
The smoothness of the eigenvectors \eqn{v_k, w_k} is induced by penalty 
matrices for both image directions, that are weighted by smoothing parameters
\eqn{\alpha_{vk}, \alpha_{wk}}. The eigenvectors \eqn{u_k} are not smoothed, 
hence the algorithm does not induce smoothness along observations.

Optimal smoothing parameters are found via a nested generalized cross 
validation. In each iteration of the TPA (tensor power algorithm), the GCV 
criterion is optimized via \code{\link[stats]{optimize}} on the interval 
specified via \code{alphaRange$v} (or \code{alphaRange$w}, respectively).

The FCP_TPA algorithm is an iterative algorithm. Convergence is assumed if 
the relative difference between the actual and the previous values are all 
below the tolerance level \code{tol}. The tolerance level is increased 
automatically, if the algorithm has not converged after \code{maxIter} steps 
and if \code{adaptTol = TRUE}. If the algorithm did not converge after
\code{maxIter} steps (or \code{2 * maxIter}) steps, the function throws a
warning.
}
\examples{
 # set.seed(1234)
 
 N <- 100
 S1 <- 75
 S2 <- 75

 # define "true" components
 v <- sin(seq(-pi, pi, length.out = S1))
 w <- exp(seq(-0.5, 1, length.out = S2))
 
 # simulate tensor data with dimensions N x S1 x S2
 X <- rnorm(N, sd = 0.5) \%o\% v \%o\% w
 
 # create penalty matrices (penalize first differences for each dimension)
 Pv <- crossprod(diff(diag(S1)))
 Pw <- crossprod(diff(diag(S2)))
 
 # estimate one eigentensor
 res <- FCP_TPA(X, K = 1, penMat = list(v = Pv, w = Pw),
             alphaRange = list(v = c(1e-4, 1e4), w = c(1e-4, 1e4)),
             verbose = TRUE)
 
 # plot the results and compare to true values
 plot(res$V)
 points(v/sqrt(sum(v^2)), pch = 20)
 legend("topleft", legend = c("True", "Estimated"), pch = c(20, 1))
 
 plot(res$W)
 points(w/sqrt(sum(w^2)), pch = 20)
 legend("topleft", legend = c("True", "Estimated"), pch = c(20, 1))
}
\references{
G. I. Allen, "Multi-way Functional Principal Components 
  Analysis", IEEE International Workshop on Computational Advances in 
  Multi-Sensor Adaptive Processing, 2013.
}
\seealso{
\code{\link{fcptpaBasis}}
}
