% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/makeRepeated.R
\name{makeRepeated}
\alias{makeRepeated}
\title{Reproduce Repeated-Measures Data from ANOVA Summary Statistics}
\usage{
makeRepeated(
  n,
  k,
  means,
  sds,
  f_stat,
  df_between = k - 1,
  df_within = (n - 1) * (k - 1),
  structure = c("cs", "ar1", "toeplitz"),
  names = paste0("time_", 1:k),
  items = 1,
  lowerbound = 1,
  upperbound = 5,
  return_corr_only = FALSE,
  diagnostics = FALSE,
  ...
)
}
\arguments{
\item{n}{Integer. Sample size used in the original study.}

\item{k}{Integer. Number of repeated measures (timepoints).}

\item{means}{Numeric vector of length \code{k}.
Mean values reported for each timepoint.}

\item{sds}{Numeric vector of length \code{k}.
Standard deviations reported for each timepoint.}

\item{f_stat}{Numeric. The reported repeated-measures ANOVA
F-statistic for the within-subjects factor.}

\item{df_between}{Degrees of freedom between conditions (default: \code{k - 1}.}

\item{df_within}{Degrees of freedom within-subjects
(default: \code{(n - 1) * (k - 1)}).}

\item{structure}{Character. Correlation structure to assume:
\code{"cs"}, \code{"ar1"}, or \code{"toeplitz"} (default = \code{"cs"}).}

\item{names}{Character vector of length \code{k}. Variable names for each
timepoint (default: \code{"time_1"} to \code{"time_k"}).}

\item{items}{Integer. Number of items used to generate each scale score
(passed to \link{lfast}).}

\item{lowerbound, }{Integer. Lower bounds for Likert-type response
scales (default: 1).}

\item{upperbound, }{Integer. upper bounds for Likert-type response
scales (default: 5).}

\item{return_corr_only}{Logical. If \code{TRUE}, return only the
estimated correlation matrix.}

\item{diagnostics}{Logical. If \code{TRUE}, include diagnostic summaries
such as feasible F-statistic range and effect sizes.}

\item{...}{Reserved for future use.}
}
\value{
A named list with components:

\describe{
\item{\code{data}}{A data frame of simulated repeated-measures responses
(unless \code{return_corr_only = TRUE}).}
\item{\code{correlation_matrix}}{The estimated inter-timepoint correlation matrix.}
\item{\code{structure}}{The correlation structure assumed.}
\item{\code{achieved_f}}{The F-statistic produced by the estimated \code{rho} value
(if \code{diagnostics = TRUE}).}
\item{\code{feasible_f_range}}{Minimum and maximum achievable F-values under the
chosen structure (shown if diagnostics are requested).}
\item{\code{recommended_f}}{Conservative, moderate, and strong F-statistic
suggestions for similar designs.}
\item{\code{effect_size_raw}}{Unstandardised effect size across timepoints.}
\item{\code{effect_size_standardised}}{Effect size standardised by average variance.}
}
}
\description{
Constructs a synthetic dataset and inter-timepoint correlation matrix
from a repeated-measures ANOVA result, based on reported means, standard
deviations, and an F-statistic. This is useful when only summary statistics
are available from published studies.
}
\details{
This function estimates the average correlation between repeated measures
by matching the reported F-statistic, under one of three assumed
correlation structures:
\itemize{
\item \code{"cs"} (\emph{Compound Symmetry}): The Default. Assumes all timepoints are
equally correlated. Common in standard RM-ANOVA settings.
\item \code{"ar1"} (\emph{First-Order Autoregressive}): Assumes correlations decay
exponentially with time lag.
\item \code{"toeplitz"} (\emph{Linearly Decreasing}): Assumes correlation declines
linearly with time lag - a middle ground between \code{"cs"} and \code{"ar1"}.
}

The function then generates a data frame of synthetic item-scale ratings
using \link{lfast}, and adjusts them to match the estimated correlation
structure using \link{lcor}.

Set \code{return_corr_only = TRUE} to extract only the estimated
correlation matrix.
}
\examples{

set.seed(42)

out1 <- makeRepeated(
  n = 64,
  k = 3,
  means = c(3.1, 3.5, 3.9),
  sds = c(1.0, 1.1, 1.0),
  items = 4,
  f_stat = 4.87,
  structure = "cs",
  diagnostics = FALSE
)

head(out1$data)
out1$correlation_matrix

out2 <- makeRepeated(
  n = 32, k = 4,
  means = c(2.75, 3.5, 4.0, 4.4),
  sds = c(0.8, 1.0, 1.2, 1.0),
  f_stat = 16,
  structure = "ar1",
  items = 5,
  lowerbound = 1, upperbound = 7,
  return_corr_only = FALSE,
  diagnostics = TRUE
)

print(out2)


out3 <- makeRepeated(
  n = 64, k = 4,
  means = c(2.0, 2.25, 2.75, 3.0),
  sds = c(0.8, 0.9, 1.0, 0.9),
  items = 4,
  f_stat = 24,
  # structure = "toeplitz",
  diagnostics = TRUE
)

str(out3)

}
\seealso{
\code{\link{lfast}}, \code{\link{lcor}}
}
