#' Undertake undertake coefficient averaging using Bayesian Model Avergaing (BMA), weighting different models by their probabailities
#'
#' @param model_table a table of competing models generated by `gam_model_probs`
#' @param terms a vector of names starting with "Intercept" plus the names of the covariates used in the GAM model (these are the names of the variables in  `data` )
#' @param thresh a probability threshold value above which to combine competing models
#' @param relative a logical to indicate whether the probabilities in `data` are relative (`Pr(M)`) or absolute (`Pr(M|D)`)
#' @param data the input data with a named Intercept term
#'
#' @return A `matrix` of the probability weighted averaged coefficient estimates from multiple models.
#' @importFrom mgcv gam
#' @importFrom stats as.formula
#' @importFrom dplyr select
#' @importFrom tidyselect all_of
#'
#' @examples
#' library(cols4all)
#' library(dplyr)
#' library(sf)
#' library(glue)
#' library(purrr)
#' library(mgcv)
#' library(sf)
#' library(ggplot2)
#' # data
#' data(productivity)
#' data = productivity |> filter(year == "1970") |> mutate(Intercept = 1)
#' # create and evaluate multiple models
#' svc_res_gam = evaluate_models(data, STVC = FALSE)
#' # determine their probabilities
#' mod_comp_svc <- gam_model_probs(svc_res_gam)
#' # combine the model coefficients
#' svc_bma <- do_bma(mod_comp_svc,
#'                   terms = c("Intercept", "unemp", "pubC"),
#'                   thresh = 0.1,
#'                   relative = FALSE,
#'                   data = data)
#' head(svc_bma)
#' # join back to spatial layer
#' data(us_data)
#' svc_bma_sf <-
#' us_data |>
#' select(GEOID) |>
#' left_join(productivity |>
#'   filter(year == "1970") |>
#'   select(GEOID, year) |>
#'   cbind(svc_bma)) |>
#'   relocate(geometry, .after = last_col())
#' # and map
#' tit =expression(paste(""*beta[`Public Capital`]*" "))
#' ggplot(data = svc_bma_sf, aes(fill=pubC)) +
#'   geom_sf() +
#'   scale_fill_continuous_c4a_div(palette="brewer.blues",name=tit) +
#'   coord_sf() +
#'   theme_void()
#' @export
do_bma = function(model_table,
                  terms,
                  thresh = 0.1,
                  relative = FALSE,
                  data){
  # 6.1 re-create the top models and extract coefficients
  extract_vcs = function(formulae, terms, data) {
    out_list = list()
    for(i in 1:length(formulae)){
      # formula and model
      f = as.formula(formulae[i])
      mod.i = gam(f, data = data)
      # VC predictions
      betas.s = calculate_vcs(mod.i, terms, data)
      out_list[[i]] <- betas.s
    }
    out_list
  }
  # 6.2 weight the coefficients
  weighted_vcs <- function(coef_list, probs, terms){
    coef_out = NULL
    for(i in paste0("b_", terms)) {
      coef_tab <- NULL
      # extract the coefficients
      for (j in 1:length(probs)) {
        coef_tab = cbind(coef_tab,
                         coef_list[[j]] |> select(all_of(i)) |> unlist())
      }
      # weighted average - BMA
      coef.i = coef_tab %*% probs
      # add to the out table
      coef_out = cbind(coef_out, coef.i |> unlist())
    }
    colnames(coef_out)  <- terms
    rownames(coef_out) <- 1:nrow(coef_out)
    coef_out
  }
  # extract the probabilities
  if(!relative) {
    n = sum(model_table$`Pr(M|D)` > 0.1)
    probs = model_table$`Pr(M|D)`[1:n]
  } else {
    n = sum(model_table$`Pr(M)` > 0.1) + 1
    probs = as.numeric(model_table$`Pr(M)`[2:n])
    # add the first one back in
    probs = c(1-probs[1], probs)
    # re-scale
    probs = probs / sum(probs)
  }
  # get the competing model VCs
  terms = c("Intercept", "unemp", "pubC")
  f = model_table$f[1:n]
  svc_list = extract_vcs(formulae = f, terms, data)
  # combine
  svc <- weighted_vcs(svc_list, probs, terms)
  svc
}
