% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/lambda_grid.R
\name{LambdaGridGraphical}
\alias{LambdaGridGraphical}
\title{Grid of penalty parameters (graphical model)}
\usage{
LambdaGridGraphical(
  xdata,
  pk = NULL,
  lambda_other_blocks = 0.1,
  K = 100,
  tau = 0.5,
  n_cat = 3,
  implementation = PenalisedGraphical,
  start = "cold",
  scale = TRUE,
  resampling = "subsampling",
  PFER_method = "MB",
  PFER_thr = Inf,
  FDP_thr = Inf,
  Lambda_cardinal = 50,
  lambda_max = NULL,
  lambda_path_factor = 0.001,
  max_density = 0.5,
  ...
)
}
\arguments{
\item{xdata}{data matrix with observations as rows and variables as columns.
For multi-block stability selection, the variables in data have to be
ordered by group.}

\item{pk}{optional vector encoding the grouping structure. Only used for
multi-block stability selection where \code{pk} indicates the number of
variables in each group. If \code{pk=NULL}, single-block stability
selection is performed.}

\item{lambda_other_blocks}{optional vector of parameters controlling the
level of sparsity in neighbour blocks for the multi-block procedure. To use
jointly a specific set of parameters for each block,
\code{lambda_other_blocks} must be set to \code{NULL} (not recommended).
Only used for multi-block stability selection, i.e. if \code{length(pk)>1}.}

\item{K}{number of resampling iterations.}

\item{tau}{subsample size. Only used if \code{resampling="subsampling"} and
\code{cpss=FALSE}.}

\item{n_cat}{number of categories used to compute the stability score.
Possible values are 2 or 3.}

\item{implementation}{function to use for graphical modelling. If
\code{implementation=PenalisedGraphical}, the algorithm implemented in
\code{\link[glassoFast]{glassoFast}} is used for regularised estimation of
a conditional independence graph. Alternatively, a user-defined function
can be provided.}

\item{start}{character string indicating if the algorithm should be
initialised at the estimated (inverse) covariance with previous penalty
parameters (\code{start="warm"}) or not (\code{start="cold"}). Using
\code{start="warm"} can speed-up the computations, but could lead to
convergence issues (in particular with small \code{Lambda_cardinal}). Only
used for \code{implementation=PenalisedGraphical} (see argument
\code{"start"} in \code{\link[glassoFast]{glassoFast}}).}

\item{scale}{logical indicating if the correlation (\code{scale=TRUE}) or
covariance (\code{scale=FALSE}) matrix should be used as input of
\code{\link[glassoFast]{glassoFast}} if
\code{implementation=PenalisedGraphical}. Otherwise, this argument must be
used in the function provided in \code{implementation}.}

\item{resampling}{resampling approach. Possible values are:
\code{"subsampling"} for sampling without replacement of a proportion
\code{tau} of the observations, or \code{"bootstrap"} for sampling with
replacement generating a resampled dataset with as many observations as in
the full sample. Alternatively, this argument can be a function to use for
resampling. This function must use arguments named \code{data} and
\code{tau} and return the IDs of observations to be included in the
resampled dataset.}

\item{PFER_method}{method used to compute the upper-bound of the expected
number of False Positives (or Per Family Error Rate, PFER). If
\code{PFER_method="MB"}, the method proposed by Meinshausen and Bühlmann
(2010) is used. If \code{PFER_method="SS"}, the method proposed by Shah and
Samworth (2013) under the assumption of unimodality is used.}

\item{PFER_thr}{threshold in PFER for constrained calibration by error
control. If \code{PFER_thr=Inf} and \code{FDP_thr=Inf}, unconstrained
calibration is used (the default).}

\item{FDP_thr}{threshold in the expected proportion of falsely selected
features (or False Discovery Proportion) for constrained calibration by
error control. If \code{PFER_thr=Inf} and \code{FDP_thr=Inf}, unconstrained
calibration is used (the default).}

\item{Lambda_cardinal}{number of values in the grid of parameters controlling
the level of sparsity in the underlying algorithm.}

\item{lambda_max}{optional maximum value for the grid in penalty parameters.
If \code{lambda_max=NULL}, the maximum value is set to the maximum
covariance in absolute value. Only used if
\code{implementation=PenalisedGraphical}.}

\item{lambda_path_factor}{multiplicative factor used to define the minimum
value in the grid.}

\item{max_density}{threshold on the density. The grid is defined such that
the density of the estimated graph does not exceed max_density.}

\item{...}{additional parameters passed to the functions provided in
\code{implementation} or \code{resampling}.}
}
\value{
A matrix of lambda values with \code{length(pk)} columns and
  \code{Lambda_cardinal} rows.
}
\description{
Generates a relevant grid of penalty parameter values for penalised graphical
models.
}
\examples{
\donttest{
# Single-block simulation
set.seed(1)
simul <- SimulateGraphical()

# Generating grid of 10 values
Lambda <- LambdaGridGraphical(xdata = simul$data, Lambda_cardinal = 10)

# Ensuring PFER < 5
Lambda <- LambdaGridGraphical(xdata = simul$data, Lambda_cardinal = 10, PFER_thr = 5)

# Multi-block simulation
set.seed(1)
simul <- SimulateGraphical(pk = c(10, 10))

# Multi-block grid
Lambda <- LambdaGridGraphical(xdata = simul$data, pk = c(10, 10), Lambda_cardinal = 10)

# Denser neighbouring blocks
Lambda <- LambdaGridGraphical(
  xdata = simul$data, pk = c(10, 10),
  Lambda_cardinal = 10, lambda_other_blocks = 0
)

# Using different neighbour penalties
Lambda <- LambdaGridGraphical(
  xdata = simul$data, pk = c(10, 10),
  Lambda_cardinal = 10, lambda_other_blocks = c(0.1, 0, 0.1)
)
stab <- GraphicalModel(
  xdata = simul$data, pk = c(10, 10),
  Lambda = Lambda, lambda_other_blocks = c(0.1, 0, 0.1)
)
stab$Lambda

# Visiting from empty to full graphs with max_density=1
Lambda <- LambdaGridGraphical(
  xdata = simul$data, pk = c(10, 10),
  Lambda_cardinal = 10, max_density = 1
)
bigblocks <- BlockMatrix(pk = c(10, 10))
bigblocks_vect <- bigblocks[upper.tri(bigblocks)]
N_blocks <- unname(table(bigblocks_vect))
N_blocks # max number of edges per block
stab <- GraphicalModel(xdata = simul$data, pk = c(10, 10), Lambda = Lambda)
apply(stab$Q, 2, max, na.rm = TRUE) # max average number of edges from underlying algo
}
}
\seealso{
Other lambda grid functions: 
\code{\link{LambdaGridRegression}()},
\code{\link{LambdaSequence}()}
}
\concept{lambda grid functions}
